//
//  PPFloatingMenuButton.m
//
//  Created by eddie on 2015/6/4.
//  Copyright (c) 2015年 Eddie. All rights reserved.
//

#import "PPFloatingMenuButton.h"
#import "PPFloatingMenuItemView.h"
#import "PPButton.h"


////////////////////////////////////////////////////////////////////////////////////////////////////

#define DEGREES_TO_RADIANS(degrees) (degrees * M_PI / 180)

////////////////////////////////////////////////////////////////////////////////////////////////////

NSInteger const PPFloatingMenuButton_ItemGap = 10;
CGFloat const PPFloatingMenuButton_AnimationDuration = 0.2;
CGSize const PPFloatingMenuButton_ShadowOffset = {-3, 5};
CGFloat const PPFloatingMenuButton_ShadowRadius = 5.0;
CGFloat const PPFloatingMenuButton_ShadowOpacity = 0.4;

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPFloatingMenuButton() <PPButtonDelegate, PPFloatingMenuItemViewDelegate>
@property (nonatomic, assign) id<PPFloatingMenuButtonDelegate> delegate;
@property (nonatomic, assign) NSInteger identifier;
@property (nonatomic, retain) PPButton *mainButton;
@property (nonatomic, retain) NSArray *mainButtonConstraints;
@property (nonatomic, retain) UIView *privMaskView;
@property (nonatomic, retain) NSArray *privMaskViewConstraints;
@property (nonatomic, retain) UITapGestureRecognizer *tapRecognizer;
@property (nonatomic, retain) NSMutableArray *itemViews;
@property (nonatomic, retain) NSArray *itemViewsConstraints;
@property (nonatomic, assign) NSInteger tappedItemIdentifier;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPFloatingMenuButton

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Lify cycle methods

//================================================================================
//
//================================================================================
- (id)initWithDelegate:(id<PPFloatingMenuButtonDelegate>)delegate identifier:(NSInteger)identifier
{
    if(self = [super init])
    {
        self.delegate = delegate;
        self.identifier = identifier;
        self.tappedItemIdentifier = PPFloatingMenuButton_ItemNone;
        self.layer.shadowOffset = PPFloatingMenuButton_ShadowOffset;
        self.layer.shadowRadius = PPFloatingMenuButton_ShadowRadius;
        self.layer.shadowOpacity = PPFloatingMenuButton_ShadowOpacity;
        
        _mainButton = [[PPButton alloc] initWithFrame:CGRectZero];
        self.mainButton.delegate = self;
        self.mainButton.translatesAutoresizingMaskIntoConstraints = NO;
        [self.mainButton addControlEvents:UIControlEventTouchDown];

        [self addSubview:self.mainButton];
        
        //////////////////////////////////////////////////
        
        self.itemViews = [NSMutableArray array];
        
        //////////////////////////////////////////////////
        
        _tapRecognizer = [[UITapGestureRecognizer alloc] initWithTarget:self action:@selector(onTapWindowMask)];
        [self.tapRecognizer setNumberOfTouchesRequired:1];

        //////////////////////////////////////////////////

        _privMaskView = [[UIView alloc] init];
        self.privMaskView.translatesAutoresizingMaskIntoConstraints = NO;
        self.privMaskView.backgroundColor = [UIColor clearColor];
        [self.privMaskView addGestureRecognizer:self.tapRecognizer];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.delegate = nil;
    self.menuItemTextFont = nil;
    self.menuItemTextColor = nil;
    self.menuItemTextBackgroundColor = nil;
    
    [self removeConstraints:self.mainButtonConstraints];
    [self.mainButton removeFromSuperview];
    self.mainButton = nil;
    self.mainButtonConstraints = nil;
    
    //////////////////////////////////////////////////
    
    for(PPFloatingMenuItemView *view in self.itemViews)
    {
        [view removeFromSuperview];
    }
    
    self.itemViews = nil;

    //////////////////////////////////////////////////
    
    [self.privMaskView removeGestureRecognizer:self.tapRecognizer];
    [self.privMaskView removeFromSuperview];
    self.privMaskView = nil;

    //////////////////////////////////////////////////
    
    self.tapRecognizer = nil;
    self.itemViewsConstraints = nil;
    self.privMaskViewConstraints = nil;
    
    [super dealloc];
}


//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
//    NSLog(@"%s", __func__);
    
    //////////////////////////////////////////////////
    // mainButton的constraint是百分比，所以只要設定一次。
    
    if(self.mainButtonConstraints == nil)
    {
        NSArray *constraints = [self mainButtonConstraintsWithParent:self];
        [self addConstraints:constraints];
        self.mainButtonConstraints = constraints;
    }

    
    //////////////////////////////////////////////////

    [super layoutSubviews];
}



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
//
//================================================================================
- (NSArray *)mainButtonConstraintsWithParent:(id)parent
{
    NSMutableArray *constraints = [NSMutableArray array];
    
    [constraints addObject:[NSLayoutConstraint constraintWithItem:self.mainButton
                                                        attribute:NSLayoutAttributeCenterX
                                                        relatedBy:NSLayoutRelationEqual
                                                           toItem:parent
                                                        attribute:NSLayoutAttributeCenterX
                                                       multiplier:1.0
                                                         constant:0.0]];
    
    [constraints addObject:[NSLayoutConstraint constraintWithItem:self.mainButton
                                                        attribute:NSLayoutAttributeCenterY
                                                        relatedBy:NSLayoutRelationEqual
                                                           toItem:parent
                                                        attribute:NSLayoutAttributeCenterY
                                                       multiplier:1.0
                                                         constant:0.0]];
    
    [constraints addObject:[NSLayoutConstraint constraintWithItem:self.mainButton
                                                        attribute:NSLayoutAttributeWidth
                                                        relatedBy:NSLayoutRelationEqual
                                                           toItem:parent
                                                        attribute:NSLayoutAttributeWidth
                                                       multiplier:1
                                                         constant:0.0]];
    
    [constraints addObject:[NSLayoutConstraint constraintWithItem:self.mainButton
                                                        attribute:NSLayoutAttributeHeight
                                                        relatedBy:NSLayoutRelationEqual
                                                           toItem:parent
                                                        attribute:NSLayoutAttributeHeight
                                                       multiplier:1
                                                         constant:0.0]];
    
    return constraints;
}


//================================================================================
//
//================================================================================
- (NSArray *)privMaskViewConstraintsWithParent:(id)parent
{
    NSMutableArray *constraints = [NSMutableArray array];
    
    [constraints addObject:[NSLayoutConstraint constraintWithItem:self.privMaskView
                                                        attribute:NSLayoutAttributeCenterX
                                                        relatedBy:NSLayoutRelationEqual
                                                           toItem:parent
                                                        attribute:NSLayoutAttributeCenterX
                                                       multiplier:1.0
                                                         constant:0.0]];
    
    [constraints addObject:[NSLayoutConstraint constraintWithItem:self.privMaskView
                                                        attribute:NSLayoutAttributeCenterY
                                                        relatedBy:NSLayoutRelationEqual
                                                           toItem:parent
                                                        attribute:NSLayoutAttributeCenterY
                                                       multiplier:1.0
                                                         constant:0.0]];
    
    [constraints addObject:[NSLayoutConstraint constraintWithItem:self.privMaskView
                                                        attribute:NSLayoutAttributeWidth
                                                        relatedBy:NSLayoutRelationEqual
                                                           toItem:parent
                                                        attribute:NSLayoutAttributeWidth
                                                       multiplier:1
                                                         constant:0.0]];
    
    [constraints addObject:[NSLayoutConstraint constraintWithItem:self.privMaskView
                                                        attribute:NSLayoutAttributeHeight
                                                        relatedBy:NSLayoutRelationEqual
                                                           toItem:parent
                                                        attribute:NSLayoutAttributeHeight
                                                       multiplier:1
                                                         constant:0.0]];
 
    return constraints;
}


//================================================================================
//
//================================================================================
- (NSArray *)itemViewsConstraintsWithParent:(UIView *)parent isFold:(BOOL)isFold
{
    CGSize parentSize = parent.bounds.size;
    CGPoint mainButtonOrigin = self.frame.origin;
    CGSize mainButtonSize = self.frame.size;
    NSMutableArray *constraints = [NSMutableArray array];
    int itemCount = (int)[self.itemViews count];
    CGPoint itemOffset = CGPointZero;
    CGFloat prevItemHeight = 0.0;
    BOOL isInTopSide = (self.center.y < self.superview.center.y);
    
    // !! 下面是判斷item在superview的左方或右方，以決定文字在左或右
    // 所以這邊的center是相對於superview的座標系，所以superview的center要校正
    CGFloat adjustedCenterX = self.superview.center.x-self.superview.frame.origin.x;
    PPFloatingMenuItemView_LayoutStyle layoutStyle = (self.center.x < adjustedCenterX) ?
                                                     PPFloatingMenuItemView_LayoutStyle_IconText :
                                                     PPFloatingMenuItemView_LayoutStyle_TextIcon;
    
    for(int i=0; i<itemCount; i++)
    {
        PPFloatingMenuItemView *itemView = [self.itemViews objectAtIndex:i];
        CGRect itemIconFrame = CGRectZero;
        CGRect itemViewFrame = CGRectZero;
        CGPoint itemViewCenter = CGPointZero;
        
        [itemView setLayoutStyle:layoutStyle];
        itemViewFrame.size = [itemView estimatedViewSize];
        itemIconFrame = [itemView estimatedIconFrame];
        itemOffset.x = -itemIconFrame.origin.x + (mainButtonSize.width - itemIconFrame.size.width) / 2;
        
        //////////////////////////////////////////////////

        if(isFold == NO)
        {
            if(isInTopSide == YES)
            {
                if(i==0)
                {
                    prevItemHeight = mainButtonSize.height;
                }
                
                itemOffset.y += (prevItemHeight + PPFloatingMenuButton_ItemGap);
                prevItemHeight = itemViewFrame.size.height;
            }
            else
            {
                itemOffset.y -= (itemViewFrame.size.height + PPFloatingMenuButton_ItemGap);
            }
        }
        
        //////////////////////////////////////////////////

        itemViewFrame.origin.x = mainButtonOrigin.x+itemOffset.x;
        
        if(CGRectGetMinX(itemViewFrame) < PPFloatingMenuButton_ItemGap)
        {
            // 左邊超出螢幕
            itemViewFrame.size.width -= (PPFloatingMenuButton_ItemGap-itemViewFrame.origin.x);
            itemViewFrame.origin.x = PPFloatingMenuButton_ItemGap;
        }
        else if(CGRectGetMaxX(itemViewFrame) > parentSize.width-PPFloatingMenuButton_ItemGap)
        {
            // 右邊超出螢幕
            itemViewFrame.size.width -= (CGRectGetMaxX(itemViewFrame)-(parentSize.width-PPFloatingMenuButton_ItemGap));
        }
        
        itemViewFrame.origin.y = mainButtonOrigin.y+itemOffset.y;
        itemViewCenter = CGPointMake(CGRectGetMidX(itemViewFrame), CGRectGetMidY(itemViewFrame));
        
        //////////////////////////////////////////////////
        // !! iOS 8.3 以後不能用指定left/top constant的方式指定位置
        //    改為設定centerX/centerY的multiplier
        
        [constraints addObject:[NSLayoutConstraint constraintWithItem:itemView
                                                            attribute:NSLayoutAttributeCenterX
                                                            relatedBy:NSLayoutRelationEqual
                                                               toItem:parent
                                                            attribute:NSLayoutAttributeCenterX
                                                           multiplier:(itemViewCenter.x*2/parentSize.width)
                                                             constant:0.0]];
        
        [constraints addObject:[NSLayoutConstraint constraintWithItem:itemView
                                                            attribute:NSLayoutAttributeCenterY
                                                            relatedBy:NSLayoutRelationEqual
                                                               toItem:parent
                                                            attribute:NSLayoutAttributeCenterY
                                                           multiplier:(itemViewCenter.y*2/parentSize.height)
                                                             constant:0.0]];
        
        [constraints addObject:[NSLayoutConstraint constraintWithItem:itemView
                                                            attribute:NSLayoutAttributeWidth
                                                            relatedBy:NSLayoutRelationEqual
                                                               toItem:parent
                                                            attribute:NSLayoutAttributeWidth
                                                           multiplier:itemViewFrame.size.width/parentSize.width
                                                             constant:0.0]];

        [constraints addObject:[NSLayoutConstraint constraintWithItem:itemView
                                                            attribute:NSLayoutAttributeHeight
                                                            relatedBy:NSLayoutRelationEqual
                                                               toItem:parent
                                                            attribute:NSLayoutAttributeHeight
                                                           multiplier:itemViewFrame.size.height/parentSize.height
                                                             constant:0.0]];
    }
    
    return constraints;
}


//================================================================================
//
//================================================================================
- (void)animateMoveMenuItemWithFold:(BOOL)isFold completion:(void (^)(void))completion
{
//    NSLog(@"1:completion(%d) self(%d)", (int)[completion retainCount], (int)[self retainCount]);

    __block typeof(self) blockSelf = self;
    __block UIView *blockMenuParentView = [self menuParentView];
    
    [UIView animateWithDuration:PPFloatingMenuButton_AnimationDuration
                     animations:^{
                         
                         //////////////////////////////////////////////////
                         // change alpha effect

                         for (PPFloatingMenuItemView *view in blockSelf.itemViews)
                         {
                             view.alpha = isFold ? 0.0 : 1.0;
                         }
                         
                         
                         //////////////////////////////////////////////////
                         // change item position
                         
                         if(blockSelf.itemViewsConstraints != nil)
                         {
                             [blockMenuParentView removeConstraints:blockSelf.itemViewsConstraints];
                         }
                         
                         NSArray *constraints = [blockSelf itemViewsConstraintsWithParent:blockMenuParentView isFold:isFold];
                         [blockMenuParentView addConstraints:constraints];
                         blockSelf.itemViewsConstraints = constraints;
                         
                         [blockMenuParentView layoutIfNeeded];
                     }
                     completion:^ (BOOL finished){
                     
                         if(completion != nil)
                         {
                             completion();
                         }
                     }];
}


//================================================================================
//
//================================================================================
- (UIView *)menuParentView
{
    return self.superview;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPButtonDelegate methods

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    if(ppButton == self.mainButton)
    {
        [self showMenu];
        
        if([self.delegate respondsToSelector:@selector(floatingMenuButton:didTapButtonWithIdentifier:)])
        {
            [self.delegate floatingMenuButton:self didTapButtonWithIdentifier:self.identifier];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UITapGestureRecognizer action

//================================================================================
//
//================================================================================
- (void)onTapWindowMask
{
    self.tappedItemIdentifier = PPFloatingMenuButton_ItemNone;
    [self hideMenu];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPFloatingMenuItemViewDelegate methods

//================================================================================
//
//================================================================================
- (void)floatingMenuItemView:(id)view didTapItemWithIdentifier:(NSInteger)identifier
{
    self.tappedItemIdentifier = identifier;
    [self hideMenu];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

//================================================================================
//
//================================================================================
- (void)setMainButtonBackgroundImage:(UIImage *)image forState:(UIControlState)state
{
    [self.mainButton setBackgroundImage:image forState:state];
    [self.mainButton setBackgroundImage:image forState:UIControlStateDisabled];
}


//================================================================================
//
//================================================================================
- (void)addMenuItemWithIdentifier:(NSInteger)identifier icon:(UIImage *)icon text:(NSString *)text
{
    PPFloatingMenuItemView *view = [[PPFloatingMenuItemView alloc] initWithIdentifier:identifier
                                                                                 icon:icon
                                                                                 text:text
                                                                             delegate:self];
    
    if(view != nil)
    {
        view.translatesAutoresizingMaskIntoConstraints = NO;
        view.menuItemTextFont = self.menuItemTextFont;
        view.menuItemTextColor = self.menuItemTextColor;
        view.menuItemTextBackgroundColor = self.menuItemTextBackgroundColor;
        view.menuItemIconSize = self.menuItemIconSize;
        
        [self.itemViews addObject:view];
        [view release];
    }
}


//================================================================================
//
//================================================================================
- (void)removeItems
{
    [self.itemViews removeAllObjects];
    
    if(self.itemViewsConstraints != nil)
    {
        [[self menuParentView] removeConstraints:self.itemViewsConstraints];
        self.itemViewsConstraints = nil;
    }
}


//================================================================================
//
//================================================================================
- (void)showMenu
{
    UIView *menuParentView = [self menuParentView];
    
    //////////////////////////////////////////////////
    
    if(menuParentView != nil && [self.itemViews count] > 0)
    {
        // MARK:點選button需要轉動效果時加上這行
//        self.transform = CGAffineTransformMakeRotation(DEGREES_TO_RADIANS(45));
        
        self.mainButton.enabled = NO;
        
        //////////////////////////////////////////////////
        
        [menuParentView addSubview:self.privMaskView];
        
        for (PPFloatingMenuItemView *view in self.itemViews)
        {
            [menuParentView addSubview:view];
            view.alpha = 0.0;
        }
        
        //////////////////////////////////////////////////
        
        NSArray *constraints = nil;
        
        constraints = [self privMaskViewConstraintsWithParent:menuParentView];
        [menuParentView addConstraints:constraints];
        self.privMaskViewConstraints = constraints;
        
        constraints = [self itemViewsConstraintsWithParent:menuParentView isFold:YES];
        [menuParentView addConstraints:constraints];
        self.itemViewsConstraints = constraints;
        
        // !! must call to let items begin in right position
        [menuParentView layoutIfNeeded];
        
        //////////////////////////////////////////////////
        
        // unfold menu items
        __block typeof(self) blockSelf = self;
        
        dispatch_async(dispatch_get_main_queue(), ^{
            
            [blockSelf animateMoveMenuItemWithFold:NO completion:nil];
        });
    }
}


//================================================================================
//
//================================================================================
- (void)hideMenu
{
    if([self menuIsHidden] == NO)
    {
        __block UIView *blockMenuParentView = [self menuParentView];
        __block typeof(self) blockSelf = self;
        
        [self animateMoveMenuItemWithFold:YES completion:^{
            
            for(PPFloatingMenuItemView *view in blockSelf.itemViews)
            {
                [view removeFromSuperview];
            }
            
            [blockSelf.privMaskView removeFromSuperview];
            
            //////////////////////////////////////////////////
            
            [blockMenuParentView removeConstraints:blockSelf.privMaskViewConstraints];
            blockSelf.privMaskViewConstraints = nil;
            
            [blockMenuParentView removeConstraints:blockSelf.itemViewsConstraints];
            blockSelf.itemViewsConstraints = nil;
  
            // 應該不用，先mark掉。
//            [blockMenuParentView layoutIfNeeded];
            
            //////////////////////////////////////////////////
            
            if([blockSelf.delegate respondsToSelector:@selector(floatingMenuButton:didTapItemWithIdentifier:)])
            {
                [blockSelf.delegate floatingMenuButton:blockSelf didTapItemWithIdentifier:blockSelf.tappedItemIdentifier];
            }
            
            //////////////////////////////////////////////////
            
            blockSelf.mainButton.enabled = YES;
            blockSelf.transform = CGAffineTransformIdentity;
        }];
    }
}


//================================================================================
//
//================================================================================
- (BOOL)menuIsHidden
{
    return (self.privMaskView == nil || self.privMaskView.superview == nil);
}



@end
