//
//  PPGoogleDriveOperation_LoadFile.m
//
//
//  Created by Mike on 13/4/15.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "PPGoogleDriveOperation_LoadFile.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPGoogleDriveOperation_LoadFile()

@interface PPGoogleDriveOperation_LoadFile()

@property (nonatomic, assign) NSUInteger fileSize;
@property (nonatomic, retain) GTMSessionFetcher *sessionFetcher;
- (void)fetcher:(GTMSessionFetcher *)fetcher finishedWithData:(NSData *)retrievedData error:(NSError *)error;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPGoogleDriveOperation_LoadFile

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize loadFile    = loadFile_;
@synthesize intoPath    = intoPath_;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (void)dealloc
{
    [self.sessionFetcher stopFetching];
    self.sessionFetcher = nil;
    
    //////////////////////////////////////////////////
    
    [loadFile_ release];
    [intoPath_ release];
    [_fileID release];
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Executing the Operation

//================================================================================
//
//================================================================================
- (void)main
{
    [self performSelectorInBackground:@selector(download) withObject:nil];
}


//================================================================================
//
//================================================================================
- (void)cancel
{
    [self.sessionFetcher stopFetching];
    
    //////////////////////////////////////////////////
    
    [super cancel];
    
    //////////////////////////////////////////////////
    
    if(self.finished==NO)
    {
        if(self.executing==YES)
        {
            [self setExecuting:NO];
        }
        
        [self setFinished:YES];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Chunk Download Method

//================================================================================
//
//================================================================================
- (void)download
{
    
    __block NSError *returnError = nil;
    
    @autoreleasepool
    {
        do
        {
            if(self.loadFile==nil || self.intoPath==nil)
            {
                returnError = PPErrorParameterInvalidity(nil);
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
          
            NSArray *loadFiles = [[GTLRDriveService sharedServiceDrive] filesForPathComponents:([self.fileID length]<=0)?[self.loadFile pathComponents]:@[[[self.loadFile pathComponents] lastObject]]
                                                                       currentPathComponentID:self.fileID
                                                                                        error:&returnError];
            if(returnError!=nil)
            {
                if(returnError.code==404)
                {
                    returnError = PPErrorMake(PPCloudCommonError_PathNotExist, @"File not found", nil);
                }
                
                break;
            }

            //////////////////////////////////////////////////

            if([loadFiles count]<1)
            {
                returnError = PPErrorMake(PPCloudCommonError_PathNotExist, @"File not found", nil);
                break;
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            if([loadFiles count]>1)
            {
                returnError = PPErrorMake(PPCloudCommonError_MultipleItemsWithSamePath, @"More than one file.", nil);
                break;
            }
            
            
            //////////////////////////////////////////////////
            
            GTLRDrive_File *loadFile = [loadFiles firstObject];
            
            //////////////////////////////////////////////////

            //目錄無法下載
            if([loadFile.mimeType isEqualToString:GTLDriveFile_MimeType_Folder]==YES)
            {
                returnError = PPErrorParameterInvalidity(nil);
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSString *downloadUrl = [NSString stringWithFormat:@"https://www.googleapis.com/drive/v3/files/%@?alt=media",
                                     loadFile.identifier];
            
            self.fileSize = [loadFile.quotaBytesUsed unsignedIntegerValue];
            
            //////////////////////////////////////////////////
  
            if(self.fileSize>PPCloud_DownloadChunkSize)
            {
                // sync
                [self chunkDownloadForUrl:downloadUrl];
            }
            else
            {
                // asncy
                [self singleFileDownloadForUrl:downloadUrl];
            }

        }
        while (0);
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        if(returnError!=nil)
        {
            if(self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppGoogleDriveOperation:loadFileFailedWithError:)]==YES)
            {
                [self.delegate ppGoogleDriveOperation:self loadFileFailedWithError:returnError];
            }
            
            [self completion];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Download Method

//================================================================================
//
//================================================================================
- (void)chunkDownloadForUrl:(NSString *)url
{
    __block typeof(self) blockSelf = self;
    __block NSError *returnError = nil;
    
    @autoreleasepool
    {
        do
        {
            NSMutableArray *chunkFileList = [NSMutableArray array];
            
            if(chunkFileList==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
      
            NSString *rangeStirng = nil;
            
            for(NSUInteger byteIndex=0; byteIndex<self.fileSize; byteIndex+=PPCloud_DownloadChunkSize)
            {
                if(byteIndex+PPCloud_DownloadChunkSize>=self.fileSize)
                {
                    rangeStirng = [NSString stringWithFormat:@"bytes=%td-",byteIndex];
                }
                else
                {
                    rangeStirng = [NSString stringWithFormat:@"bytes=%td-%td",byteIndex,byteIndex+PPCloud_DownloadChunkSize-1];
                }
                
                //////////////////////////////////////////////////
                
                if([rangeStirng length]<=0)
                {
                    returnError = PPErrorParameterInvalidity(nil);
                    break;
                }
                
                //////////////////////////////////////////////////
                
                NSString *chunkFilePath = [self.intoPath stringByReplacingOccurrencesOfString:[self.intoPath stringByDeletingPathExtension]
                                                                                   withString:[[self.intoPath stringByDeletingPathExtension] stringByAppendingString:rangeStirng]];
                
                if([chunkFilePath length]<=0)
                {
                    returnError = PPErrorParameterInvalidity(nil);
                    break;
                }
                
                [chunkFileList addObject:chunkFilePath];
                
                //////////////////////////////////////////////////
                
                self.sessionFetcher = [GTMSessionFetcher fetcherWithURLString:url];
                
                if(self.sessionFetcher ==nil)
                {
                    returnError = PPErrorOperationFailed(nil);
                    break;
                }
                
                //////////////////////////////////////////////////

                self.sessionFetcher.authorizer             = [GTLRDriveService sharedServiceDrive].authorizer;
                self.sessionFetcher.destinationFileURL     = [NSURL fileURLWithPath:chunkFilePath];
                
                //////////////////////////////////////////////////
                
                [self.sessionFetcher setRequestValue:rangeStirng
                                  forHTTPHeaderField:@"Range"];
                
                //////////////////////////////////////////////////
                
           
                self.sessionFetcher.downloadProgressBlock  = ^(int64_t bytesWritten, int64_t totalBytesWritten, int64_t totalBytesExpectedToWrite) {
                    if(blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(ppGoogleDriveOperation:loadProgress:forFile:)]==YES)
                    {
                        [blockSelf.delegate ppGoogleDriveOperation:blockSelf
                                                      loadProgress:(float)(totalBytesWritten+byteIndex)/(float)blockSelf.fileSize
                                                           forFile:blockSelf.intoPath];
                    }
                };
                
                
                ////////////////////////////////////////////////////////////////////////////////////////////////////
                
                dispatch_group_t group = dispatch_group_create();
                
                if(group==NULL)
                {
                    returnError = PPErrorMake(PPCloudCommonError_Unknown, @"dispatch_group_create 失敗", nil);
                    break;
                }
                
                dispatch_group_enter(group);
                
                [self.sessionFetcher beginFetchWithCompletionHandler:^(NSData * _Nullable data, NSError * _Nullable error) {
                    
                    returnError = error;
                    
                    dispatch_group_leave(group);
                    
                }];
                
                dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
                
                dispatch_release(group);
                
                //////////////////////////////////////////////////
                
                //下載過程發生錯誤，不在下載
                if(returnError!=nil)
                {
                    break;
                }
                
                // for loop end
            }
        
            //////////////////////////////////////////////////
            
            if(returnError!=nil)
            {
                break;
            }
            
            // 把下載的檔案內容讀取，組成NSData, 寫入成完成檔案
            [self combineChunkFileList:chunkFileList
                 toDestinationFilePath:self.intoPath
                             withError:&returnError];
          
        }
        while (0);
        
        //////////////////////////////////////////////////

        [returnError retain];
        
        //////////////////////////////////////////////////
        
        dispatch_async(dispatch_get_main_queue(), ^{
            [blockSelf fetcher:blockSelf.sessionFetcher finishedWithData:nil error:returnError];
            [returnError release];
        });
        
        // auto release end
    }
}






//================================================================================
//
//================================================================================
- (void)singleFileDownloadForUrl:(NSString *)url
{
    @autoreleasepool
    {
        NSError *returnError = nil;
        
        do
        {
            if([url length]<=0)
            {
                returnError = PPErrorOperationFailed(nil);
                break;
            }
            
            //////////////////////////////////////////////////
            
            self.sessionFetcher = [GTMSessionFetcher fetcherWithURLString:url];
            
            if(self.sessionFetcher ==nil)
            {
                returnError = PPErrorOperationFailed(nil);
                break;
            }
            
            //////////////////////////////////////////////////
            
            self.sessionFetcher.authorizer             = [GTLRDriveService sharedServiceDrive].authorizer;
            self.sessionFetcher.destinationFileURL     = [NSURL fileURLWithPath:self.intoPath];
            
            //////////////////////////////////////////////////
            
            __block typeof(self) blockSelf = self;
            
            self.sessionFetcher.downloadProgressBlock  = ^(int64_t bytesWritten, int64_t totalBytesWritten, int64_t totalBytesExpectedToWrite) {
                if(blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(ppGoogleDriveOperation:loadProgress:forFile:)]==YES)
                {
                    [blockSelf.delegate ppGoogleDriveOperation:blockSelf
                                                  loadProgress:(float)totalBytesWritten/(float)blockSelf.fileSize
                                                       forFile:blockSelf.intoPath];
                }
            };
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            [self.sessionFetcher beginFetchWithCompletionHandler:^(NSData * _Nullable data, NSError * _Nullable error) {
                [self fetcher:blockSelf.sessionFetcher finishedWithData:data error:error];
            }];
        }
        while (0);
        
        // auto release end
        
        if(returnError!=nil)
        {
            [self fetcher:self.sessionFetcher finishedWithData:nil error:returnError];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Download Call Back Method

//================================================================================
//
//================================================================================
- (void)fetcher:(GTMSessionFetcher *)fetcher finishedWithData:(NSData *)retrievedData error:(NSError *)error
{
    if(error!=nil)
    {
        if(self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppGoogleDriveOperation:loadFileFailedWithError:)]==YES)
        {
            [self.delegate ppGoogleDriveOperation:self loadFileFailedWithError:error];
        }
    }
    else
    {
        if(self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppGoogleDriveOperation:loadedFile:)]==YES)
        {
            [self.delegate ppGoogleDriveOperation:self loadedFile:self.intoPath];
        }
    }
    
    //////////////////////////////////////////////////
    
    [self completion];
}

@end
