//
//  PPDropboxOperation_LoadFile.m
//
//
//  Created by Mike on 13-3-14.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "PPDropboxOperation_LoadFile.h"

#import "DBTransportBaseClient+Internal.h"

@interface PPDropboxOperation_LoadFile ()
@property (atomic,retain) DBDownloadDataTask *downloadTask;

@end
@implementation PPDropboxOperation_LoadFile

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize loadFile    = loadFile_;
@synthesize intoPath    = intoPath_;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (void)dealloc
{
    [_downloadTask cancel];
    [_downloadTask release];
    _downloadTask = nil;
    
    //////////////////////////////////////////////////
    
    [loadFile_ release];
    [intoPath_ release];
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Executing the Operation

//================================================================================
//
//================================================================================
- (void)main
{
    // 取一次檔案大小
    if(self.fileByteSize<=0)
    {
        __block NSError *returnError = nil;
        
        do
        {
            DBFILESUserAuthRoutes *fileRoutes = [DBClientsManager authorizedClient].filesRoutes;
            
            if(fileRoutes==nil)
            {
                returnError = PPErrorMake(PPCloudCommonError_Unknown, @"fileRoutes 失敗", nil);
                [returnError retain];
                
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                returnError = PPErrorMake(PPCloudCommonError_Unknown, @"dispatch_group_create 失敗", nil);
                [returnError retain];
                
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            __block typeof(self) blockself = self ;
            
            [[fileRoutes getMetadata:self.loadFile] setResponseBlock:^(DBFILESMetadata * _Nullable filesMetadata, DBFILESGetMetadataError * _Nullable metadataError, DBRequestError * _Nullable dbRequestError)
             {
                 if(dbRequestError!=nil)
                 {
                     if([DBTransportBaseClient statusCodeIsRouteError:[[dbRequestError statusCode] intValue]]==YES)
                     {
                         returnError = PPErrorMake(PPCloudCommonError_PathNotExist, @"Deleted file path not found", nil);
                         [returnError retain];
                     }
                     else
                     {
                         returnError = dbRequestError.nsError;
                         [returnError retain];
                     }
                 }
                 else
                 {
                     if([filesMetadata respondsToSelector:@selector(size)]==YES)
                     {
                         blockself.fileByteSize = [[filesMetadata performSelector:@selector(size)] integerValue];
                     }
                 }
                
                 //////////////////////////////////////////////////
                 
                 dispatch_group_leave(group);
             }];
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            dispatch_release(group);
            
            //////////////////////////////////////////////////
            
            //發生錯誤，就不用下載
            if(returnError!=nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            if(self.fileByteSize>PPCloud_DownloadChunkSize)
            {
                [self chunkDownload];
            }
            else
            {
                [self download];
            }
        }
        while (0);
        
        //////////////////////////////////////////////////
        
        if(returnError!=nil)
        {
            [self performSelectorOnMainThread:@selector(sendFailDelegateWithError:)
                                   withObject:returnError
                                waitUntilDone:NO];
            
            [returnError release];
        }
    }
    else if(self.fileByteSize>PPCloud_DownloadChunkSize)
    {
        [self chunkDownload];
    }
    else
    {
        [self download];
    }
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Canceling Operations

//================================================================================
//
//================================================================================
- (void)cancel
{
    [self.downloadTask cancel];
    
    //////////////////////////////////////////////////
    
    [super cancel];
    
    //////////////////////////////////////////////////

    if(self.finished==NO)
    {
        if(self.executing==YES)
        {
            [self setExecuting:NO];
        }
        
        [self setFinished:YES];
    } 
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Download Method

//================================================================================
//
//================================================================================
- (void)chunkDownload
{
    __block NSError *returnError = nil;
    
    do
    {
        if(self.loadFile==nil || self.intoPath==nil)
        {
            returnError = PPErrorParameterInvalidity(nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        DBFILESUserAuthRoutes *fileRoutes = [DBClientsManager authorizedClient].filesRoutes;
        
        if(fileRoutes==nil)
        {
            returnError  = PPErrorParameterInvalidity(nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSMutableArray *chunkFileList = [NSMutableArray array];
        
        if(chunkFileList==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSUInteger endByteIndex = 0;
        
        NSString *rangeStirng = nil;
        
        for(NSUInteger beginByteIndex=0; beginByteIndex<self.fileByteSize; beginByteIndex+=PPCloud_DownloadChunkSize)
        {
            if(beginByteIndex+PPCloud_DownloadChunkSize>=self.fileByteSize)
            {
                rangeStirng = [NSString stringWithFormat:@"bytes=%td-",beginByteIndex];
                
                endByteIndex = self.fileByteSize;
            }
            else
            {
                rangeStirng = [NSString stringWithFormat:@"bytes=%td-%td",beginByteIndex,beginByteIndex+PPCloud_DownloadChunkSize-1];
                
                endByteIndex = beginByteIndex+PPCloud_DownloadChunkSize-1;
            }
            
            //////////////////////////////////////////////////
            
            if([rangeStirng length]<=0)
            {
                returnError = PPErrorParameterInvalidity(nil);
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSString *chunkFilePath = [self.intoPath stringByReplacingOccurrencesOfString:[self.intoPath stringByDeletingPathExtension]
                                                                               withString:[[self.intoPath stringByDeletingPathExtension] stringByAppendingString:rangeStirng]];
            
            if([chunkFilePath length]<=0)
            {
                returnError = PPErrorParameterInvalidity(nil);
                break;
            }
            
            [chunkFileList addObject:chunkFilePath];
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                returnError = PPErrorMake(PPCloudCommonError_Unknown, @"dispatch_group_create 失敗", nil);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            __block typeof(self) blockself = self;
            
            self.downloadTask = [fileRoutes downloadData:self.loadFile
                                         byteOffsetStart:@(beginByteIndex)
                                           byteOffsetEnd:@(endByteIndex)];
            
            [[self.downloadTask setResponseBlock:^(DBFILESFileMetadata *fileMetadata, DBFILESDownloadError *routeError, DBRequestError *error, NSData *fileContents) {
                
                if(fileMetadata)
                {
                    if([fileContents writeToFile:chunkFilePath atomically:YES]==NO)
                    {
                        returnError = PPErrorMake(PPCloudCommonError_WriteFileFail, @"WriteFileFail", nil);
                    }
                    
                    dispatch_group_leave(group);
                }
                else
                {
                    if([DBTransportBaseClient statusCodeIsRouteError:[[error statusCode] intValue]]==YES)
                    {
                        returnError = PPErrorMake(PPCloudCommonError_PathNotExist, @"Deleted file path not found", nil);
                        [returnError retain];
                    }
                    else
                    {
                        returnError = [blockself convertErrorFromSystemError:error];
                    }
                   
                    dispatch_group_leave(group);
                }
            }]
             setProgressBlock:^(int64_t bytesDownloaded, int64_t totalBytesDownloaded, int64_t totalBytesExpectedToDownload)
             {
                 if([blockself.delegate respondsToSelector:@selector(ppDropboxOperation:loadProgress:forFile:)]==YES)
                 {
                     CGFloat progress = (CGFloat)(totalBytesDownloaded+beginByteIndex)/blockself.fileByteSize;
                     
                     [blockself.delegate ppDropboxOperation:self loadProgress:progress forFile:blockself.intoPath];
                 }
             }];
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            dispatch_release(group);
            
            //////////////////////////////////////////////////
            
            //下載過程發生錯誤，不在下載
            if(returnError!=nil)
            {
                break;
            }
            
            
            // for loop end
        }
        
        //////////////////////////////////////////////////
        
        if(returnError!=nil)
        {
            break;
        }
        
        // 把下載的檔案內容讀取，組成NSData, 寫入成完成檔案
        [self combineChunkFileList:chunkFileList
             toDestinationFilePath:self.intoPath
                         withError:&returnError];
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(returnError!=nil)
    {
        [returnError retain];
        
        [self performSelectorOnMainThread:@selector(sendFailDelegateWithError:)
                               withObject:[returnError autorelease]
                            waitUntilDone:NO];
    }
    else
    {
        [self performSelectorOnMainThread:@selector(sendSuccessDelegate)
                               withObject:nil
                            waitUntilDone:NO];
    }
}


//================================================================================
//
//================================================================================
- (void)download
{
    NSError *returnError = nil;
    
    do
    {
        if(self.loadFile==nil || self.intoPath==nil)
        {
            returnError = PPErrorParameterInvalidity(nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        DBFILESUserAuthRoutes *fileRoutes = [DBClientsManager authorizedClient].filesRoutes;
        
        if(fileRoutes==nil)
        {
            returnError  = PPErrorParameterInvalidity(nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block typeof(self) blockself = self;
        
        self.downloadTask = [fileRoutes downloadData:self.loadFile];
        
        [[self.downloadTask setResponseBlock:^(DBFILESFileMetadata *fileMetadata, DBFILESDownloadError *routeError, DBRequestError *error, NSData *fileContents) {
            
            if(fileMetadata)
            {
                [fileContents writeToFile:blockself.intoPath atomically:YES];
                
                [blockself performSelectorOnMainThread:@selector(sendSuccessDelegate)
                                            withObject:nil
                                         waitUntilDone:NO];
            }
            else
            {
                NSError *convertedError = nil;
                
                if([DBTransportBaseClient statusCodeIsRouteError:[[error statusCode] intValue]]==YES)
                {
                    convertedError = PPErrorMake(PPCloudCommonError_PathNotExist, @"Deleted file path not found", nil);
                    
                }
                else
                {
                    convertedError = [blockself convertErrorFromSystemError:error];
                }
                
                [blockself performSelectorOnMainThread:@selector(sendFailDelegateWithError:)
                                            withObject:convertedError
                                         waitUntilDone:NO];
            }
        }]
         setProgressBlock:^(int64_t bytesDownloaded, int64_t totalBytesDownloaded, int64_t totalBytesExpectedToDownload)
         {
             if([blockself.delegate respondsToSelector:@selector(ppDropboxOperation:loadProgress:forFile:)]==YES)
             {
                 CGFloat progress = (CGFloat)totalBytesDownloaded/totalBytesExpectedToDownload;
                 
                 [blockself.delegate ppDropboxOperation:self loadProgress:progress forFile:blockself.intoPath];
             }
         }];
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(returnError!=nil)
    {
        [self performSelectorOnMainThread:@selector(sendFailDelegateWithError:)
                               withObject:returnError
                            waitUntilDone:NO];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Send Delegate Method

//================================================================================
//
//================================================================================
- (void)sendSuccessDelegate
{
    //    __block typeof(self) blockself = self;
    //
    //    dispatch_async(dispatch_get_main_queue(), ^{
    if([self.delegate respondsToSelector:@selector(ppDropboxOperation:loadedFile:contentType:)]==YES)
    {
        [self.delegate ppDropboxOperation:self
                               loadedFile:self.intoPath
                              contentType:[self.loadFile pathExtension]];
    }
    
    //////////////////////////////////////////////////
    
    [self completion];
    //    });
}


//================================================================================
//
//================================================================================
- (void)sendFailDelegateWithError:(NSError *)error
{
    if(error!=nil &&
       [self.delegate respondsToSelector:@selector(ppDropboxOperation:loadFileFailedWithError:)]==YES)
    {
        [self.delegate
         ppDropboxOperation:self
         loadFileFailedWithError:error];
    }
    
    [self completion];
}
@end
