//
//  PPHDRGenerator.m
//  PPCameraView
//
//  Created by sanhue cheng on 2020/4/17.
//

#import "PPHDRGenerator.h"
#import "UIImage+MTLTexture.h"
#import <MetalKit/MTKTextureLoader.h>
#import "UIImage+BITMAPPTR.h"
#import "MetalHelper.h"

#import "PPImageAlignment.h"
#import "MetalFunctionCreateMapping.h"
#import "MetalFunctionAlignmentOffset.h"
#import "MetalFunctionGenerateHDR.h"


////////////////////////////////////////////////////////////////////////////////////////////////////
@interface PPHDRGenerator ()
@property (nonatomic, retain) id<MTLDevice> mtlDevice;
@property (nonatomic, retain) id<MTLLibrary> library;
@property (nonatomic, retain) id <MTLCommandQueue> commandQueue;
@property (nonatomic, retain) MTKTextureLoader *loader; // 纹理加载器

// computing pipeline
@property (nonatomic, retain) id <MTLComputePipelineState> computingPipelineState;

// 原圖紋理
@property (nonatomic, retain) NSMutableArray <id<MTLTexture>> *imageTextures;

// 纹理缓存
@property (nonatomic, assign) CVMetalTextureCacheRef textureCache;

@property (nonatomic, retain) MetalFunctionGenerateHDR *generateHDR;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation PPHDRGenerator





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - init


//==============================================================================
//
//==============================================================================
- (instancetype)initWithDevice:(id<MTLDevice>)device
{
    self = [super init];
    
    if(self)
    {
        self.mtlDevice = device;
        NSString *metallibPath = [[NSBundle mainBundle] pathForResource:@"PPHDRGenerator" ofType:@"metallib"];
        self.library = [[self.mtlDevice newLibraryWithFile:metallibPath error:nil] autorelease];
        self.commandQueue = [[self.mtlDevice newCommandQueue] autorelease]; // 获取一个渲染队列，其中装载需要渲染的指令 MTLCommandBuffer
        
        //////////////////////////////////////////////////        
//        id <MTLFunction> imageInvertFunction = [self.library newFunctionWithName:@"imageInvert"];
//        self.computingPipelineState = [self.mtlDevice newComputePipelineStateWithFunction:imageInvertFunction error:nil];
        
        //////////////////////////////////////////////////
        
        self.loader = [[[MTKTextureLoader alloc] initWithDevice:self.mtlDevice] autorelease];
        self.imageTextures = [NSMutableArray array];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    self.loader = nil;
    for (id<MTLTexture> texture in self.imageTextures)
    {
        [texture setPurgeableState:MTLPurgeableStateVolatile];
    }
    [self.imageTextures removeAllObjects];
    self.imageTextures = nil;
    
    self.generateHDR = nil;
    self.computingPipelineState = nil;
    self.commandQueue = nil;
    self.library = nil;
    self.mtlDevice = nil;
    
    //////////////////////////////////////////////////
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private image



//==============================================================================
//
//==============================================================================
- (id<MTLTexture>)textureFromImage:(CPImage *)image
{
    if(image==nil)
    {
        return nil;
    }

    id<MTLTexture> texture = nil;

    @autoreleasepool {
        // MARK:若要檢查aligment過程的數值是否和android一樣，要用bgraData。
//        NSData *imageData = [image bgraData];
        
        // MARK:單純HDR用jpg data即可，速度較快。

        NSData *imageData = UIImageJPEGRepresentation(image, 0.8);

        if(imageData==nil)
        {
            return nil;
        }

        NSError* err = nil;
        texture = [self.loader newTextureWithData:imageData options:@{} error:&err];
    }
    return [texture autorelease];
}


//==============================================================================
//
//==============================================================================
- (void)loadTexturesFromImages:(NSArray <id<MTLTexture>>*)images
{
    for (id<MTLTexture> texture in self.imageTextures)
    {
        [texture setPurgeableState:MTLPurgeableStateVolatile];
    }
    [self.imageTextures removeAllObjects];
    
    for (id<MTLTexture> image in images)
    {
        [self.imageTextures addObject:image];
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private encoder



//==============================================================================
//
//==============================================================================
- (void)encodeImageInvertToCommandBuffer:(id<MTLCommandBuffer>)commandBuffer
                           sourceTexture: (nonnull id <MTLTexture>) sourceTexture
                      destinationTexture: (nonnull id <MTLTexture>) destinationTexture
{
//    NSLog(@"%s in", __PRETTY_FUNCTION__);
    id<MTLComputeCommandEncoder> computingCommandEncoder = [commandBuffer computeCommandEncoder]; // 通过渲染描述符构建 encoder
        
    [computingCommandEncoder setComputePipelineState:self.computingPipelineState];
    [computingCommandEncoder setTexture:sourceTexture atIndex:0];
    [computingCommandEncoder setTexture:destinationTexture atIndex:1];

    //////////////////////////////////////////////////

    // Calculate a threadgroup size.
    NSUInteger w = self.computingPipelineState.threadExecutionWidth;
    NSUInteger h = self.computingPipelineState.maxTotalThreadsPerThreadgroup / w;
    MTLSize threadsPerThreadgroup = MTLSizeMake(w, h, 1);
    MTLSize threadgroupsPerGrid = MTLSizeMake((sourceTexture.width + w - 1) / w,
                                              (sourceTexture.height + h - 1) / h,
                                              1);
    
    // Encode the compute command.
    [computingCommandEncoder dispatchThreadgroups: threadgroupsPerGrid
                          threadsPerThreadgroup: threadsPerThreadgroup];
    
    // End the compute pass.
    [computingCommandEncoder endEncoding];
    
//    NSLog(@"%s out", __PRETTY_FUNCTION__);
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private (alignment)

//==============================================================================
// 亮度中位數
//==============================================================================
- (LuminanceInfo *)computeMedianLuminanceWithTexture:(id<MTLTexture>)texture
                                                rect:(CGRect)rect
                                          imageWidth:(NSInteger)imageWidth
{
    if (texture==nil)
    {
        return nil;
    }
    
    int mtb_x = rect.origin.x;
    int mtb_y = rect.origin.y;
    int mtb_width = rect.size.width;
    int mtb_height = rect.size.height;
    
    int n_samples_c = 100;
    int n_w_samples = (int)sqrt(n_samples_c);
    int n_h_samples = n_samples_c / n_w_samples;
    int histo[256] = {0};
    
    int *imageDataBytes = malloc(texture.width*texture.height*sizeof(int));
    if(imageDataBytes == nil)
    {
        return nil;
    }
    
    [texture getBytes:imageDataBytes
          bytesPerRow:texture.width*sizeof(int)
           fromRegion:MTLRegionMake2D(0, 0, texture.width, texture.height)
          mipmapLevel:0];
    
    //////////////////////////////////////////////////
    
    // 清空
    for (int i = 0; i < 256; i++)
    {
        histo[i] = 0;
    }
    
    int total = 0;
    for (int y = 0; y < n_h_samples; y++)
    {
        double alpha = ((double) y + 1.0) / ((double) n_h_samples + 1.0);
        int y_coord = mtb_y + (int) (alpha * mtb_height);
        
        for (int x = 0; x < n_w_samples; x++)
        {
            double beta = ((double) x + 1.0) / ((double) n_w_samples + 1.0);
            int x_coord = mtb_x + (int) (beta * mtb_width);
            
            NSInteger dataIndex = y_coord*imageWidth+x_coord;
            int color = imageDataBytes[dataIndex];
            
            int r = (color & 0xFF0000) >> 16;
            int g = (color & 0xFF00) >> 8;
            int b = (color & 0xFF);
            int luminance = MAX(r, g);
            luminance = MAX(luminance, b);
            histo[luminance]++;
            total++;
        }
    }
    
    free(imageDataBytes);
    
    //////////////////////////////////////////////////
    
    int middle = total / 2;
    int count = 0;
    BOOL noisy = NO;
    
    for (int i = 0; i < 256; i++)
    {
        count += histo[i];
        
        if (count >= middle)
        {
            int noise_threshold = 4;
            int n_below = 0;
            
            for (int j = 0; j <= i - noise_threshold; j++)
            {
                n_below += histo[j];
            }
            double frac_below = n_below / (double) total;
            if (frac_below < 0.2)
            {
                noisy = true;
            }
            return [LuminanceInfo luminanceInfoWithMedianValue:i noisy:noisy];
        }
    }
    
    return [LuminanceInfo luminanceInfoWithMedianValue:127 noisy:YES];
}


//==============================================================================
//
//==============================================================================
- (void)alignmentWithImageSize:(CGSize)imageSize offsetsX:(int *)offsets_x offsetsY:(int *)offsets_y
{
    int full_width = imageSize.width;
    int full_height = imageSize.height;
    int mtb_width = full_width / 2;
    int mtb_height = full_height / 2;
    int mtb_x = mtb_width / 2;
    int mtb_y = mtb_height / 2;
    
    
    NSMutableArray <LuminanceInfo *>*luminanceInfos = [NSMutableArray array];
    NSMutableArray <id<MTLTexture>> *mappingTableTextures = [NSMutableArray array];
    
    //////////////////////////////////////////////////
    // 建立mapping table
    
    MetalFunctionCreateMapping *createMapping = [[[MetalFunctionCreateMapping alloc] initWithDevice:self.mtlDevice libray:self.library] autorelease];
    
    for (NSInteger i=0; i < [self.imageTextures count]; i++)
    {
        // 計算luminance
        id <MTLTexture> sourceTexture = [self.imageTextures objectAtIndex:i];
        LuminanceInfo *luminanceInfo = [self computeMedianLuminanceWithTexture:sourceTexture
                                                                          rect:CGRectMake(mtb_x, mtb_y, mtb_width, mtb_height)
                                                                    imageWidth:imageSize.width];
        [luminanceInfos addObject:luminanceInfo];
        
        if (luminanceInfos[i].noisy)
        {
            //            mtb_allocations[i] = null;
            continue;
        }
        
        // for test
//        NSLog(@"medianValue:%d", luminanceInfo.medianValue);

        //////////////////////////////////////////////////
        
        // 建立command buffer
        id<MTLCommandBuffer> commandBuffer = nil;
        commandBuffer = [self.commandQueue commandBuffer]; // 获取一个可用的命令 buffer
        commandBuffer.label = @"CreateMapping";
                
        // 輸出結果
        id<MTLTexture> mappingTableTexture = [MetalHelper textureWithSize:CGSizeMake(mtb_width, mtb_height)
                                                              pixelFormat:sourceTexture.pixelFormat
                                                                   device:self.mtlDevice];
        [mappingTableTextures addObject:mappingTableTexture];

        [createMapping encodeToCommandBuffer:commandBuffer
                           withStartPosition:CGPointMake(mtb_x, mtb_y)
                                stopPosition:CGPointMake(mtb_x+mtb_width, mtb_y+mtb_height)
                                      median:luminanceInfo.medianValue
                               sourceTexture:sourceTexture
                               resultTexture:mappingTableTexture
                                   imageSize:imageSize];

        [commandBuffer commit];
        [commandBuffer waitUntilCompleted];
        
        // for test
//        [self dumpTexture:mappingTableTexture];
    }

    
    //////////////////////////////////////////////////
    // 算出計算offset要切成幾個block
    
    int max_dim = MAX(full_width, full_height);
    int max_ideal_size = max_dim / 150;
    int initial_step_size = 1;
    
    // 次方?
    while (initial_step_size < max_ideal_size) {
        initial_step_size *= 2;
    }
        
    //////////////////////////////////////////////////
    // 計算offset
    
    MetalFunctionAlignmentOffset *alignmentOffset = [[[MetalFunctionAlignmentOffset alloc] initWithDevice:self.mtlDevice libray:self.library] autorelease];
    id <MTLTexture> baseTexture = nil;
    
    if([mappingTableTextures count]>=2)
    {
        baseTexture = [mappingTableTextures objectAtIndex:1];
    }

    for (int i=0; i < [self.imageTextures count]; i++)
    {
        if (i == 1)
        {
            continue;
        }

        id <MTLTexture> compareTexture = [mappingTableTextures objectAtIndex:i];
        int pixel_step = 1;
        int step_size = initial_step_size;
        
        while (step_size > 1)
        {
            step_size /= 2;
            
            int pixel_step_size = step_size * pixel_step;
            
            if (pixel_step_size > mtb_width || pixel_step_size > mtb_height)
            {
                pixel_step_size = step_size;
            }
            
            //////////////////////////////////////////////////
            
            // 建立command buffer
            id<MTLCommandBuffer> commandBuffer = nil;
            commandBuffer = [self.commandQueue commandBuffer]; // 获取一个可用的命令 buffer
            commandBuffer.label = @"AlignmentOffset";
            
            int numberOfErrors = 9;
            id<MTLBuffer> resultBuffer = [[self.mtlDevice newBufferWithLength:sizeof(int)*numberOfErrors options:MTLResourceStorageModeShared] autorelease];
                        
            [alignmentOffset encodeToCommandBuffer:commandBuffer
                                 withStartPosition:CGPointMake(0, 0)
                                      stopPosition:CGPointMake(mtb_width/pixel_step_size, mtb_height/pixel_step_size)
                                    offsetPosition:CGPointMake(offsets_x[i], offsets_y[i])
                                          stepSize:pixel_step_size
                                       baseTexture:baseTexture
                                    compareTexture:compareTexture
                                      resultBuffer:resultBuffer
                                         imageSize:CGSizeMake(mtb_width, mtb_height)];

            [commandBuffer commit];
            [commandBuffer waitUntilCompleted];

            //////////////////////////////////////////////////

            int *errors = resultBuffer.contents;


            // for test (用CPU模擬GPU計算，結果不一樣，要查原因)
//            int tempBuffer[9] = {0,0,0,0,0,0,0,0,0};
//             [self testAlignmentWithStart:CGPointMake(0, 0)
//                                     stop:CGPointMake(mtb_width/pixel_step_size, mtb_height/pixel_step_size)
//                                   offset:CGPointMake(offsets_x[i], offsets_y[i])
//                                imageSize:CGSizeMake(mtb_width, mtb_height)
//                                 stepSize:pixel_step_size
//                              baseTexture:baseTexture
//                           compareTexture:compareTexture
//                             resultBuffer:tempBuffer];
//
//            errors = tempBuffer;


            
            int best_error = -1;
            int best_id = -1;
            
            for (int j = 0; j < 9; j++)
            {
                int this_error = errors[j];
                
                // for test
//                NSLog(@"result %d = %d", j, this_error);
                
                if (best_id == -1 || this_error < best_error)
                {
                    best_error = this_error;
                    best_id = j;
                }
            }
            
            if (best_error >= 2000000000)
            {
                best_id = 4;
            }
            
            if (best_id != -1)
            {
                int this_off_x = best_id % 3;
                int this_off_y = best_id / 3;
                this_off_x--;
                this_off_y--;
                offsets_x[i] += this_off_x * step_size;
                offsets_y[i] += this_off_y * step_size;
            }
            
            [resultBuffer setPurgeableState:MTLPurgeableStateVolatile];

            // for test
//            NSLog(@"offsets_x[%d] = %d", i, offsets_x[i]);
//            NSLog(@"offsets_y[%d] = %d", i, offsets_y[i]);
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)testAlignmentWithStart:(CGPoint)start
                          stop:(CGPoint)stop
                        offset:(CGPoint)offset
                     imageSize:(CGSize)imageSize
                      stepSize:(int)stepSize
                   baseTexture:(id <MTLTexture>)baseTexture
                compareTexture:(id <MTLTexture>)compareTexture
                  resultBuffer:(int *)resultBuffer
{
    for(int y=0; y<imageSize.height; y++)
    {
        for(int x=0; x<imageSize.width; x++)
        {
            int step_size = stepSize;
            int off_x = offset.x;
            int off_y = offset.y;

            x *= step_size;
            y *= step_size;

            if( x+off_x >= step_size &&
               x+off_x < imageSize.width-step_size &&
               y+off_y >= step_size &&
               y+off_y < imageSize.height-step_size)
            {
                int rValue0 = [self valueWithTexture:baseTexture x:x y:y];
                int c=0;

                for(int dy=-1;dy<=1;dy++)
                {
                    for(int dx=-1;dx<=1;dx++)
                    {
                        int rValue1 = [self valueWithTexture:compareTexture x:x+off_x+dx*step_size y:y+off_y+dy*step_size];

                        if( rValue0 != rValue1)
                        {
                            if( rValue0 != 187 && rValue1 != 187 )
                            {
                                resultBuffer[c]++;
                            }
                        }
                        
//                        NSLog(@"rValue0=%d, rValue1=%d, c=%d", rValue0, rValue1, c);
                        c++;
                    }
                }
            }
        }
    }
}


//==============================================================================
//
//==============================================================================
- (int)valueWithTexture:(id <MTLTexture>)texture x:(int)x y:(int)y
{
    int length = (int)texture.width*(int)texture.height;
    int *dataBytes = malloc(length*sizeof(int));
    
    [texture getBytes:dataBytes
          bytesPerRow:texture.width*sizeof(int)
           fromRegion:MTLRegionMake2D(0, 0, texture.width, texture.height)
          mipmapLevel:0];
    
    int value = dataBytes[x+y*texture.width];
    value = (value & 0xFF0000) >> 16;

    free(dataBytes);
    
    return value;
}


//==============================================================================
// 亮度中位數
//==============================================================================
- (void)dumpTexture:(id<MTLTexture>)texture
{
    NSUInteger length = texture.width*texture.height;
    int *dataBytes = malloc(length*sizeof(int));
    
    [texture getBytes:dataBytes
          bytesPerRow:texture.width*sizeof(int)
           fromRegion:MTLRegionMake2D(0, 0, texture.width, texture.height)
          mipmapLevel:0];
    
    for(NSUInteger i=0; i<length; i+=1000)
    {
        int value = dataBytes[i];
        
        int r = (value & 0xFF0000) >> 16;
        int g = (value & 0xFF00) >> 8;
        int b = (value & 0xFF);

        NSLog(@"%td (%d, %d, %d)", i, r, g, b);
    }
    
    free(dataBytes);
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private (hdr)

//==============================================================================
//
//==============================================================================
- (double)averageRGBWithColor:(int)color
{
    int r = (color & 0xFF0000) >> 16;
    int g = (color & 0xFF00) >> 8;
    int b = (color & 0xFF);
    return (r + g + b) / 3.0;
}


//==============================================================================
//
//==============================================================================
- (BOOL)calculateHdrParamWithTexture1:(id<MTLTexture>)texture1
                             texture2:(id<MTLTexture>)texture2
                              offsetX:(int)offset_x
                              offsetY:(int)offset_y
                           parameterA:(float *)parameterA
                           parameterB:(float *)parameterB

{
   
    //////////////////////////////////////////////////
    //  對應Android的createFunctionFromBitmaps
    //////////////////////////////////////////////////
    
    int n_samples_c = 100;
    int n_w_samples = (int)sqrt(n_samples_c);
    int n_h_samples = n_samples_c / n_w_samples;
    double avg_in = 0.0;
    double avg_out = 0.0;
    
    int *image1DataBytes = malloc(texture1.width*texture1.height*sizeof(int));
    if(image1DataBytes == nil)
    {
        return NO;
    }
    
    [texture1 getBytes:image1DataBytes
          bytesPerRow:texture1.width*sizeof(int)
           fromRegion:MTLRegionMake2D(0, 0, texture1.width, texture1.height)
          mipmapLevel:0];

    int *image2DataBytes = malloc(texture2.width*texture2.height*sizeof(int));
    if(image2DataBytes == nil)
    {
        free(image1DataBytes);
        return NO;
    }
    
    [texture2 getBytes:image2DataBytes
          bytesPerRow:texture2.width*sizeof(int)
           fromRegion:MTLRegionMake2D(0, 0, texture2.width, texture2.height)
          mipmapLevel:0];

    //////////////////////////////////////////////////
    
    NSMutableArray *x_samples = [NSMutableArray array];
    NSMutableArray *y_samples = [NSMutableArray array];
    NSMutableArray *weights = [NSMutableArray array];

    for (int y = 0; y < n_h_samples; y++)
    {
        double alpha = ((double) y + 10.0) / ((double) n_h_samples *3);
        int y_coord = (int) (alpha * texture1.width);
        
        for (int x = 0; x < n_w_samples; x++)
        {
            double beta = ((double) x + 1.0) / ((double) n_w_samples + 1.0);
            int x_coord = (int) (beta * texture1.width);
            if (x_coord + offset_x < 0 || x_coord + offset_x >= texture1.width || y_coord + offset_y < 0 || y_coord + offset_y >= texture1.height)
            {
                continue;
            }
            
            NSInteger dataIndex1 = (y_coord+offset_y)*texture1.width+(x_coord+offset_x);
            int in_col = image1DataBytes[dataIndex1];

            NSInteger dataIndex2 = y_coord*texture2.width+x_coord;
            int out_col = image2DataBytes[dataIndex2];

            double in_value = [self averageRGBWithColor:in_col];
            double out_value = [self averageRGBWithColor:out_col];
            avg_in += in_value;
            avg_out += out_value;
            [x_samples addObject:@(in_value)];
            [y_samples addObject:@(out_value)];
        }
    }
    
    free(image1DataBytes);
    free(image2DataBytes);
    
    if(x_samples.count == 0)
    {
        double in_value = 255.0;
        double out_value = 255.0;
        avg_in += in_value;
        avg_out += out_value;
        [x_samples addObject:@(in_value)];
        [y_samples addObject:@(out_value)];
    }
    
    avg_in /= x_samples.count;
    avg_out /= y_samples.count;
    
    bool is_dark_exposure = avg_in < avg_out;
    
    double min_value = [x_samples[0] doubleValue];
    double max_value = [x_samples[0] doubleValue];
    for (int i = 1; i < x_samples.count; i++)
    {
        double value = [x_samples[i] doubleValue];
        if (value < min_value)
            min_value = value;
        if (value > max_value)
            max_value = value;
    }
    
    double med_value = 0.5 * (min_value + max_value);
    
    double min_value_y = [y_samples[0] doubleValue];
    double max_value_y = [y_samples[0] doubleValue];
    for (int i = 1; i < y_samples.count; i++)
    {
        double value = [y_samples[i] doubleValue];
        if (value < min_value_y)
            min_value_y = value;
        if (value > max_value_y)
            max_value_y = value;
    }
    
    double med_value_y = 0.5 * (min_value_y + max_value_y);
    for (int i = 0; i < x_samples.count; i++)
    {
        double value = [x_samples[i] doubleValue];
        double value_y = [y_samples[i] doubleValue];
        double weight = (value <= med_value) ? value - min_value : max_value - value;
        if (is_dark_exposure)
        {
            double weight_y = (value_y <= med_value_y) ? value_y - min_value_y : max_value_y - value_y;
            if (weight_y < weight)
            {
                weight = weight_y;
            }
        }
            
        [weights addObject:@(weight)];
    }
    
    
    //////////////////////////////////////////////////
    //  對應Android的ResponseFunction建構子
    //////////////////////////////////////////////////

    if((x_samples.count != y_samples.count) ||
       (x_samples.count != weights.count) ||
       (x_samples.count <= 3))
    {
        return NO;
    }

    bool done = false;
    double sum_wx = 0.0;
    double sum_wx2 = 0.0;
    double sum_wxy = 0.0;
    double sum_wy = 0.0;
    double sum_w = 0.0;
    
    for (int i = 0; i < x_samples.count; i++)
    {
        double x = [x_samples[i] doubleValue];
        double y = [y_samples[i] doubleValue];
        double w = [weights[i] doubleValue];
        sum_wx += w * x;
        sum_wx2 += w * x * x;
        sum_wxy += w * x * y;
        sum_wy += w * y;
        sum_w += w;
    }
    
    double A_numer = sum_wy * sum_wx - sum_w * sum_wxy;
    double A_denom = sum_wx * sum_wx - sum_w * sum_wx2;

    if (ABS(A_denom) >= 1.0e-5)
    {
        *parameterA = (float) (A_numer / A_denom);
        *parameterB = (float) ((sum_wy - (*parameterA) * sum_wx) / sum_w);
        
        if (*parameterA >= 1.0e-5 && *parameterB >= 1.0e-5)
        {
            done = true;
        }
    }
    
    if (!done)
    {
        double numer = 0.0;
        double denom = 0.0;
        
        for (int i = 0; i < x_samples.count; i++)
        {
            double x = [x_samples[i] doubleValue];
            double y = [y_samples[i] doubleValue];
            double w = [weights[i] doubleValue];
            numer += w * x * y;
            denom += w * x * x;
        }
        
        if (denom < 1.0e-5)
        {
            *parameterA = 1.0f;
        }
        else
        {
            *parameterA = (float) (numer / denom);
            
            if (*parameterA < 1.0e-5)
            {
                *parameterA = 1.0e-5f;
            }
        }
        
        *parameterB = 0.0f;
    }
    return YES;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Public methods



//==============================================================================
//
//==============================================================================
- (CPImage *)hdrImageByImages:(NSArray <id<MTLTexture>>*)images
{
//    NSLog(@"%s in", __PRETTY_FUNCTION__);
    
    // for test MARK:比對android的HDR結果
//    UIImage *image0 = [UIImage imageNamed:@"photo1.jpg"];
//    UIImage *image1 = [UIImage imageNamed:@"photo0.jpg"];
//    UIImage *image2 = [UIImage imageNamed:@"photo2.jpg"];
//    images = @[image0, image1, image2];

    
    CPImage *resultImage = nil;
    id<MTLTexture> resultTexture = nil;
    //////////////////////////////////////////////////
    // 1. 影像讀取到textture

    [self loadTexturesFromImages:images];

    do {
        // 如果張數不夠，回傳中間那一張，如果只有一張就取第一張
        if([self.imageTextures count] < 3)
        {
            if([self.imageTextures count]==2)
            {
                resultTexture = [[self.imageTextures objectAtIndex:1] retain];
            }
            else if([self.imageTextures count]==1)
            {
                resultTexture = [[self.imageTextures firstObject] retain];
            }
            break;
        }
        
        id<MTLTexture> baseTexture = [self.imageTextures objectAtIndex:1];
        CGSize baseSize = CGSizeMake(baseTexture.width, baseTexture.height);
                
        //////////////////////////////////////////////////
        // 2. 計算aligment對齊需要的offset
        
        int offsetsX[3] = {0, 0, 0};
        int offsetsY[3] = {0, 0, 0};
        
        // !! aligment計算有問題需再調整，但Bracket拍照一般不會有偏移，所以先不計算。
        //    [self alignmentWithImageSize:baseSize offsetsX:offsetsX offsetsY:offsetsY];
        
        
        //////////////////////////////////////////////////
        // 3. hdr
        
        float parameterA[3] = {1.0, 1.0, 1.0};
        float parameterB[3] = {0.0, 0.0, 0.0};
        
        if([self calculateHdrParamWithTexture1:self.imageTextures[0]
                                      texture2:self.imageTextures[1]
                                       offsetX:offsetsX[0]
                                       offsetY:offsetsY[0]
                                    parameterA:&parameterA[0]
                                    parameterB:&parameterB[0]] == NO)
        {
            // 無法計算，回傳中間那一張
            resultTexture = [[self.imageTextures objectAtIndex:1] retain];
            break;
        }
        
        if([self calculateHdrParamWithTexture1:self.imageTextures[2]
                                      texture2:self.imageTextures[1]
                                       offsetX:offsetsX[2]
                                       offsetY:offsetsY[2]
                                    parameterA:&parameterA[2]
                                    parameterB:&parameterB[2]] == NO)
        {
            // 無法計算，回傳中間那一張
            resultTexture = [[self.imageTextures objectAtIndex:1] retain];
            break;
        }
        
        // 建立command buffer
        id<MTLCommandBuffer> commandBuffer = nil;
        commandBuffer = [self.commandQueue commandBuffer]; // 获取一个可用的命令 buffer
        commandBuffer.label = @"GenerateHDR";
        
        [resultTexture setPurgeableState:MTLPurgeableStateVolatile];
        [resultTexture release];
        resultTexture = [[MetalHelper textureWithSize:baseSize pixelFormat:baseTexture.pixelFormat device:self.mtlDevice] retain];
        
        if(self.generateHDR==nil)
        {
            self.generateHDR = [[[MetalFunctionGenerateHDR alloc] initWithDevice:self.mtlDevice libray:self.library] autorelease];
        }
        
        [self.generateHDR encodeToCommandBuffer:commandBuffer
                                    withOffsetX:offsetsX
                                        offsetY:offsetsY
                                     parameterA:parameterA
                                     parameterB:parameterB
                                 sourceTexture0:self.imageTextures[0]
                                 sourceTexture1:self.imageTextures[1]
                                 sourceTexture2:self.imageTextures[2]
                                  resultTexture:resultTexture
                                      imageSize:baseSize];
        
        [commandBuffer commit];
        [commandBuffer waitUntilCompleted];
        
        //    NSLog(@"%s after commit", __PRETTY_FUNCTION__);

    } while (0);
    
    
    // mark release texture
    for (id<MTLTexture> texture in self.imageTextures)
    {
        [texture setPurgeableState:MTLPurgeableStateVolatile];
    }
    [self.imageTextures removeAllObjects];

    //////////////////////////////////////////////////
    // 轉UIImage
    BOOL isRGBA = YES;
    if(resultTexture.pixelFormat==MTLPixelFormatBGRA8Unorm||
       resultTexture.pixelFormat==MTLPixelFormatBGRA8Unorm_sRGB)
    {
        isRGBA = NO;
    }
    
    resultImage = [UIImage imageFromTexture:resultTexture isRGBA:isRGBA] ;
    
    //////////////////////////////////////////////////
    // mark release resultTexture
    [resultTexture setPurgeableState:MTLPurgeableStateVolatile];
    [resultTexture release];

    return resultImage;
}

@end
