//
//  MetalFunctionGenerateHDR.m
//  PPCameraView
//
//  Created by sanhue cheng on 2020/6/2.
//

#import "MetalFunctionGenerateHDR.h"
#import "PPCVKernelTypes.h"


@interface MetalFunctionGenerateHDR ()

@property (nonatomic, strong) id<MTLDevice> mtlDevice;
@property (nonatomic, strong) id <MTLComputePipelineState> computingPipelineState;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation MetalFunctionGenerateHDR


//==============================================================================
//
//==============================================================================
- (nullable instancetype)initWithDevice:(nonnull id <MTLDevice>)device libray:(nonnull id<MTLLibrary>)library
{
    self = [super init];
    if (self)
    {
        self.mtlDevice = device;
        
        // compute pipline
        id <MTLFunction> function = [library newFunctionWithName:@"generateHDR"];
        self.computingPipelineState = [[self.mtlDevice newComputePipelineStateWithFunction:function error:nil] autorelease];
        [function release];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    self.mtlDevice = nil;
    self.computingPipelineState = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}


//==============================================================================
//
//==============================================================================
- (void)encodeToCommandBuffer:(nonnull id <MTLCommandBuffer>)commandBuffer
                  withOffsetX:(int *)offsetX
                      offsetY:(int *)offsetY
                   parameterA:(float *)parameterA
                   parameterB:(float *)parameterB
               sourceTexture0:(id<MTLTexture>)sourceTexture0
               sourceTexture1:(id<MTLTexture>)sourceTexture1
               sourceTexture2:(id<MTLTexture>)sourceTexture2
                resultTexture:(id<MTLTexture>)resultTexture
                    imageSize:(CGSize)imageSize
{
    id<MTLComputeCommandEncoder> computingCommandEncoder = [commandBuffer computeCommandEncoder]; // 通过渲染描述符构建 encoder
    id<MTLBuffer> generateHDRParams = [self.mtlDevice newBufferWithLength:sizeof(PPCVGenerateHDRParams) options:MTLResourceStorageModeShared];
    PPCVGenerateHDRParams *param = (PPCVGenerateHDRParams *)generateHDRParams.contents;
    
    for(int i=0; i<3; i++)
    {
        param->offsetX[i] = offsetX[i];
        param->offsetY[i] = offsetY[i];
        param->parameterA[i] = parameterA[i];
        param->parameterB[i] = parameterB[i];
    }

    [computingCommandEncoder setComputePipelineState:self.computingPipelineState];
    [computingCommandEncoder setTexture:sourceTexture0 atIndex:0];
    [computingCommandEncoder setTexture:sourceTexture1 atIndex:1];
    [computingCommandEncoder setTexture:sourceTexture2 atIndex:2];
    [computingCommandEncoder setTexture:resultTexture atIndex:3];
    [computingCommandEncoder setBuffer:generateHDRParams offset:0 atIndex:0];

    [generateHDRParams release];
    
    //////////////////////////////////////////////////
    // Calculate a threadgroup size.
    
    NSUInteger w = self.computingPipelineState.threadExecutionWidth;
    NSUInteger h = self.computingPipelineState.maxTotalThreadsPerThreadgroup / w;
    MTLSize threadsPerThreadgroup = MTLSizeMake(w, h, 1);
    MTLSize threadgroupsPerGrid = MTLSizeMake((imageSize.width + w - 1) / w,
                                              (imageSize.height + h - 1) / h,
                                              1);
    
    // Encode the compute command.
    [computingCommandEncoder dispatchThreadgroups: threadgroupsPerGrid
                            threadsPerThreadgroup: threadsPerThreadgroup];
    
    // End the compute pass.
    [computingCommandEncoder endEncoding];
}

@end
