//
//  NSError+Custom.h
//  
//
//  Created by Mike on 13/4/26.
//  Modified by Egg on 2015/01/22.
//  
//

#import <Foundation/Foundation.h>

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPErrorMake(va_code, va_reason, va_object)  [NSError errorWithDomain:NSStringFromClass([self class]) code:(va_code) lineNumber:(__LINE__) localizedFailureReason:(va_reason) object:(va_object)]
#define PPErrorOperationCancel(va_object)           PPErrorMake((NSErrorCustom_Code_OperationCancel), (@"Operation Cancel"), (va_object))
#define PPErrorOperationFailed(va_object)           PPErrorMake((NSErrorCustom_Code_OperationFailed), (@"Operation Failed"), (va_object))
#define PPErrorParameterInvalidity(va_object)       PPErrorMake((NSErrorCustom_Code_ParameterInvalidity), (@"Parameter Invalidity"), (va_object))

/*!
 *  Check if NSError object was allocated, make error with lineNumber & error messages,
 *  Basically makes **error easily.
 *
 *  @param va_error          Error with double pointer.
 *  @param va_checkHasError  Stop making error when *error already exist.
 *  @param va_throwWhenDebug Throw exception if YES & in DEBUG mode.
 *  @param va_object         Object in userInfo for flexibility.
 *  @param va_args           Error message with format.
 *  @param ...               Error message arguments.
 *
 */
#define PPErrorWithError(va_error, va_checkHasError, va_throwWhenDebug, va_object, va_args, ...)  [NSError makeError:va_error withDomain:NSStringFromClass([self class]) code:(NSErrorCustom_Code_ParameterInvalidity) lineNumber:(__LINE__) object:(va_object) checkHasError:va_checkHasError throwWhenDebug:va_throwWhenDebug format:(va_args), ##__VA_ARGS__]

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_ENUM(NSUInteger, NSErrorCustom_Code)
{
    NSErrorCustom_Code_OperationCancel = 0,
	NSErrorCustom_Code_OperationFailed,
	NSErrorCustom_Code_ParameterInvalidity,
    NSErrorCustom_Code_UserDefine,
};

////////////////////////////////////////////////////////////////////////////////////////////////////

extern NSString * const NSErrorCustom_Key_LineNumber;
extern NSString * const NSErrorCustom_Key_Object;

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface NSError (Custom)

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods

/*!
 *  Common error generater
 *
 *  @param domain         Error Domain
 *  @param code           Erorr code.
 *  @param lineNumber     Line Number of error happened.
 *  @param localizedFailureReason  Error reason.
 *  @param object         Object in userInfo for flexibility.
 *  @return NSError object
 */
+ (NSError *)errorWithDomain:(NSString *)domain
                        code:(NSInteger)code
                  lineNumber:(NSUInteger)lineNumber
      localizedFailureReason:(NSString *)localizedFailureReason
                      object:(id)object;


/**
 *  Check if NSError object was allocated, make error with lineNumber & error messages,
 *  Basically makes **error easily.
 *
 *  @param error          Error with double pointer.
 *  @param domain         Error Domain
 *  @param code           Erorr code.
 *  @param lineNumber     Line Number of error happened.
 *  @param object         Object in userInfo for flexibility.
 *  @param checkHasError  Stop making error when *error already exist.
 *  @param throwWhenDebug Throw exception if YES & in DEBUG mode.
 *  @param format         Error message with format & arguments.
 @return YES, make error successfully
 */
+ (BOOL)makeError:(NSError **)error
       withDomain:(NSString *)domain
             code:(NSInteger)code
       lineNumber:(NSUInteger)lineNumber
           object:(id)object
    checkHasError:(BOOL)checkHasError
   throwWhenDebug:(BOOL)throwWhenDebug
           format:(NSString *)format, ...;




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

/**
 Short message for displaying error in alert.

 @return Alert message string.
 */
- (NSString *)alertMessage;


/**
 找出指定條件的錯誤

 @param condictionDict @{DomainName:@[Code1, Code2...]}
 @return 符合條件的error
 */
- (NSError *)findErrorWithConditionDict:(NSDictionary *)condictionDict;


/**
 找出是否有網路錯誤

 @return 網路錯誤的error
 */
- (NSError *)findNetworkError;


/**
 找出是否有儲存空間錯誤

 @return 儲存空間錯誤error
 */
- (NSError *)findStorageError;


/**
 可顯示google用data包裝的錯誤資訊

 @return 資訊字串
 */
- (NSString *)googleDescription;


/**
 完整的錯誤資訊描述（包含google的data訊息）

 @return detail description
 */
- (NSString *)detailDescription;

@end
