//
//  WCTRestClientController.h
//
//
//  Created by Howard on 2016/3/9.
//  Copyright © 2016年 penpower. All rights reserved.
//

#import <Foundation/Foundation.h>

// Define
#import "WCTRestClientController+ErrorCodeDefine.h"
#import "WCTRestClientController+SettingsKey.h"

// Category
#import "WCTRestClient+Test.h"
#import "WCTRestClient+Account.h"
#import "WCTRestClient+Category.h"
#import "WCTRestClient+Contact.h"
#import "WCTRestClient+PrivateContact.h"
#import "WCTRestClient+PublicContact.h"
#import "WCTRestClient+LoginOut.h"
#import "WCTRestClient+StatisticInfo.h"
#import "WCTRestClient+Notice.h"
#import "WCTRestClient+NormalUserSetting.h"
#import "WCTRestClient+Search.h"
#import "WCTRestClient+Sync.h"
#import "WCTRestClient+System.h"
#import "WCTRestClient+MultiLayerCategory.h"
#import "WCTRestClient+Register.h"
#import "WCTRestClient+Subscribe.h"
#import "WCTRestClient+CRM.h"
#import "WCTRestClient+MyCard.h"


// Model
#import "PPVersionInfo.h"

extern NSString *const WCTRestClientController_LogoutNotification ;
extern NSString *const WCTRestClientController_LimitedAccountStateChangedNotification ;

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCTRestClientController : NSObject

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property

@property (nonatomic,assign) BOOL enableLocalMode;

/// 登入的server版號
@property (atomic, retain) PPVersionInfo *serverVersionInfo;
@property (atomic, retain) PPVersionInfo *clientSupportedVersionInfo;
@property (atomic, retain) PPVersionInfo *maxBackwardCompatibleVersionInfo;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method

+ (WCTRestClientController *)shareRestClientController;
+ (NSString *)curLoginToken;
+ (NSString *)curAccountGuid;
+ (WCTRestClient *)curRestClient;


/**
 * 從restkit回傳的Error中取得response
 * @param error RestKit的Error
 * @return NSHTTPURLResponse, 如果不存在回傳nil
 */
+ (NSHTTPURLResponse *)responseFromAFRKNetworkingError:(NSError *)error;

/**
 * 從restkit回傳的Error中取得status code
 * @param error RestKit的Error
 * @return NSInteger status code from server, 不存在則回傳NSNotFound
 */
+ (NSInteger)statusCodeFromAFRKNetworkingError:(NSError *)error;

/**
 * 從restkit回傳的Error中取得domain
 * @param error RestKit的Error
 * @return NSString Domain name
 */
+ (NSString *)domainFromAFRKNetworkingError:(NSError *)error;

/**
 * 從登出通知中取得error
 * @param notification 登出通知
 * @return NSError 造成登出的錯誤
 */
+ (NSError *)errorFromLogoutNotification:(NSNotification *)notification;


/**
 * 產生一個登出的通知
 */
+ (void)sendLogoutNotification;



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method (Server restore related)


/**
 * 是否需要還原
 * 收到登入成功的delegate後，用來判斷server是否有還原，
 * @param databaseCreateTime 這次登入時取得的資料庫建立時間，取loginInfo.databaseCreateTime
 * @return BOOL YES, if server has restored, else NO
 */
+ (BOOL)isServerRestoredWithDatabaseCreateTime:(NSDate *)databaseCreateTime;


/**
 * 紀錄這次資料庫建立時間,
 * 不用還原或處理完還原流程才紀錄時間，不然如果還原中斷，下次登入會出現
 * @param databaseCreateTime 這次登入時取得的資料庫建立時間，取loginInfo.databaseCreateTime
 */
+ (void)recordDatabaseCreateTime:(NSDate *)databaseCreateTime;




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class Method (Account subscription status)
/**
 * 目前登入帳號是否為暫停使用
 * 登入成功後才會更新值，登入前取得的值為上次登入的狀態
 * @return BOOL 帳號是否為暫停使用
 */
+ (BOOL)isLimitedAccount;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method (log related)

// 取得log path
+ (NSString *)logDirPath;



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method (Network Related)

/**
 *  清除 cached
 *  !! 清除動作很花時間，請審慎評估後再呼叫。
 */
+ (void)cleanCachedResponses;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance For Category Method


/**
 檢查連線是否過期
 
 @param error 回傳錯誤
 @return YES, 合法
 */
- (BOOL)checkTimeExpiredAndKeepConnectionWithError:(NSError **)error;



/**
 Log 方法
 
 @param format Log 訊息
 */
- (void)logMessageWithStartTime:(NSDate *)startTime format:(NSString *)format, ...;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method


/**
 *  自動登入，只限變更密碼後使用，以取得新的token
 *
 *  @param error      回傳錯誤
 *
 *  @return 自動登入 結果
 */
- (BOOL)autoLoginWithError:(NSError **)error;

/**
 *  改變URL Address
 *
 *  @param urlAddress 新的連線 Address
 *  @param error      回傳錯誤
 *
 *  @return 改變URL Address 結果
 */
- (BOOL)changeURLAddress:(NSString *)urlAddress withError:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance RestClient Operaiton Method

/**
 *  取消所有 RestKit operaiton
 */
- (void)cancellAllRestOperation;


/**
 *  等待所有 RestKit operation finished
 */
- (void)waitAllRestOperaitonAreFinished;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Login Method

/**
 *  登入
 *
 *  @param loginRequest 登入參數
 *  @param error        回傳錯誤
 *
 *  @return 結果
 */
- (WCTRCLoginResponseResult *)loginWithLoginRequest:(WCTRCLoginRequestObject *)loginRequest withError:(NSError **)error;


/**
 *  登出
 *
 *  @param error 回傳錯誤
 *
 *  @return 結果
 */
- (WCTRCLogoutResponseResult *)logoutWithError:(NSError **)error;

/**
 *  離線登入要call這個
 */
- (void)offlineLogin;


/**
 *  Token 更新
 *
 *  @param error 回傳錯誤
 *
 *  @return 結果
 */
- (WCTRCTokenInfoResponseResult *)refreshTokenWithError:(NSError **)error;


/**
 *  檢查伺服器是否合法
 *
 *  @param loginCode            檢查物件
 *  @param error                回傳錯誤
 *
 *  @return YES 合法
 */
- (BOOL)checkServerIslegalWithLoginCode:(WCTRCLoginCode *)loginCode
                                  error:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Account Method

/**
 *  取得 帳號縮圖
 *
 *  @param error 回傳錯誤
 *
 *  @return 帳號縮圖
 */
- (WCTRCInLineModel *)accountLogoWithError:(NSError **)error;


/**
 *  重設密碼
 *
 *  @param password 密碼
 *  @param error    回傳錯誤
 *
 *  @return 重設密碼結果
 */
- (WCTRCResetPasswordResponseResult *)resetPasswordWithPassword:(NSString *)password
                                                          error:(NSError **)error;

/**
 *  修改密碼
 *
 *  @param oldPassword 舊密碼
 *  @param newPassword  新密碼
 *  @param error        回傳錯誤
 *
 *  @return 修改密碼結果
 */
- (WCTRCChangePasswordResponseResult *)changePasswordWithOldPassword:(NSString *)oldPassword
                                                         newPassword:(NSString *)newPassword
                                                               error:(NSError **)error;


/**
 *  上傳帳號縮圖
 *
 *  @param image    圖像
 *  @param fileName 檔案名稱
 *  @param mineType 圖像類型 WCTRC_MineType_JPG or WCTRC_MineType_PNG
 *  @param error    回傳錯誤
 *
 *  @return 上傳帳號縮圖結果
 */
- (WCTRCAccountUploadLogoResponseResult *)uploadAccountLogoWithImage:(CPImage *)image
                                                            fileName:(NSString *)fileName
                                                            mineType:(NSString *)mineType
                                                               error:(NSError **)error;


/**
 *  取得帳號內所有資訊
 *
 *  @param error 回傳錯誤
 *
 *  @return 取得帳號內所有資訊
 */
- (WCTRCAccountInfoResponseResult *)accountInfoWithError:(NSError **)error;


/**
 *  取得其他帳號內些許資訊
 *
 *  @param error 回傳錯誤
 *
 *  @return 取得帳號內所有資訊
 */
- (WCTRCOtherAccountListResponseResult *)otherAccountsInfoWithError:(NSError **)error;


/**
 *  取得上司帳號資訊
 *
 *  @param error    回傳錯誤
 *
 *  @return 取得上司帳號資訊結果
 */
- (WCTRCAccountInfoResponseResult *)accountBossWithError:(NSError **)error;


/**
 *  取得所有下屬帳號資訊
 *
 *  @param error 回傳錯誤
 *
 *  @return 取得所有下屬帳號資訊結果
 */
- (WCTRCAccountInfoResponseResult *)accountSubordinatesWithError:(NSError **)error;


/**
 *  變更使用者名稱
 *
 *  @param displayName 新增名稱
 *  @param error 回傳錯誤
 *
 *  @return 取得所有下屬帳號資訊結果
 */
- (WCTRCMessageResponseResult *)changeDisplayName:(NSString *)displayName withError:(NSError **)error;


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Category Method

/**
 *  建立類別
 *
 *  @param categoryName 類別名稱
 *  @param clientCreatedCategoryGuid 類別 Guid
 *  @param error                       回傳錯誤
 *
 *  @return 建立類別結果
 */
- (WCTRCCategoryCreateResponseResult *)createCategoryWithCategoryName:(NSString *)categoryName
                                            clientCreatedCategoryGuid:(NSString *)clientCreatedCategoryGuid
                                                                error:(NSError **)error DEPRECATED_MSG_ATTRIBUTE("createMultiLayerCategoryWithCreationRequest:error:, WCT ServerVersion 4.0.0");

/**
 *  刪除類別
 *
 *  @param categoryGuids 刪除類別 guids ( The NSString of array)
 *  @param error         回傳錯誤
 *
 *  @return 刪除類別結果
 */
- (WCTRCCategoryDeleteResponseResult *)deleteCategoryWithCategoryGuids:(NSArray *)categoryGuids
                                                                 error:(NSError **)error DEPRECATED_MSG_ATTRIBUTE("deleteMultiLayerCategoryWithCategoryGuid:error:, WCT ServerVersion 4.0.0");


/**
 *  更新類別
 *
 *  @param newCategoryName 類別名稱
 *  @param categoryGuid    類別Guid
 *  @param error         回傳錯誤
 *
 *  @return 更新類別結果
 */
- (WCTRCCategoryUpdateResponseResult *)updateCategoryWithNewCategoryName:(NSString *)newCategoryName
                                                            categoryGuid:(NSString *)categoryGuid
                                                                   error:(NSError **)error DEPRECATED_MSG_ATTRIBUTE("updateMultiLayerCategoryWithGuid:categoryName:error:, WCT ServerVersion 4.0.0");

/**
 *  取得帳號類別列表
 *
 *  @param error 回傳錯誤
 *
 *  @return 帳號類別列表
 */
- (WCTRCCategoryInfoListResponseResult *)categoryListWithError:(NSError **)error DEPRECATED_MSG_ATTRIBUTE("multiLayerCategoryListWithError, WCT ServerVersion 4.0.0");


/**
 *  更新類別順序
 *
 *  @param request 包含順序及時間
 *  @param error 回傳錯誤
 *
 *  @return 更新類別順序結果
 */
- (WCTRCDateTimeResponseResult *)categorySetOrderWithRequest:(WCTRCCategoryOrderInfoRequest *)request
                                                       error:(NSError **)error DEPRECATED_MSG_ATTRIBUTE("setOrderToggledMultiLayerCategoryListWithRequest:error, WCT ServerVersion 4.0.0");


/**
 *  取得server上的類別順序
 *
 *  @param error 回傳錯誤
 *
 *  @return 更新類別順序結果
 */
- (WCTRCCategoryOrderInfoResponseResult *)categoryGetOrderWithError:(NSError **)error DEPRECATED_MSG_ATTRIBUTE("toggledMultiLayerCategoryListWithError:, WCT ServerVersion 4.0.0");





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Contact Method



/**
 匯出指定的contact到Salesforce
 
 @param contactList 名片guids
 @param target WCTRC_SalesforceTarget_Contact, WCTRC_SalesforceTarget_Lead
 @param salesforceAccountId guid
 @param error 回傳錯誤
 @return 匯出指定的contact到Salesforce 結果
 */
- (WCTRCMessageResponseResult *)exportToSalesforceWithContactList:(WCTRCListRequest *)contactList
                                                           target:(NSString *)target
                                              salesforceAccountId:(NSString *)salesforceAccountId
                                                            error:(NSError **)error;



/**
 匯出指定的contact到Exchange server
 
 @param contactList 名片guids
 @param error 回傳錯誤
 @return 匯出指定的contact到Exchange server 結果
 */
- (WCTRCMessageResponseResult *)exportToExchangeServerWithContactList:(WCTRCListRequest *)contactList
                                                                error:(NSError **)error;
////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Private Contact Method


/**
 *  私有名片類別操作
 *
 *  @param contactSetCategoriesRequest 需求物件
 *  @param error                       回傳錯誤
 *
 *  @return 私有名片類別操作結果
 */
- (WCTRCPrivateContactOperateCategoryResponseResult *)privateContactOperateCategoryWithContactSetCategoriesRequest:(WCTRCContactSetCategoriesRequest *)contactSetCategoriesRequest
                                                                                                             error:(NSError **)error;


/**
 *  私有名片類別操作
 *
 *  @param contactSetCategoriesRequest 需求物件
 *   @param progressHandler            回報處理進度
 *  @param error                       回傳錯誤
 *
 *  @return 私有名片類別操作結果
 */
- (WCTRCPrivateContactOperateCategoryResponseResult *)privateContactOperateCategoryWithContactSetCategoriesRequest:(WCTRCContactSetCategoriesRequest *)contactSetCategoriesRequest
                                                                                                   progressHandler:(void(^)(CGFloat progress))progressHandler
                                                                                                             error:(NSError **)error;

/**
 *  上傳私有類別區名片縮圖
 *
 *  @param image    圖像
 *  @param fileName 檔案名稱
 *  @param mineType 圖像類型
 *  @param error    回傳錯誤
 *
 *  @return 上傳私有類別區名片縮圖結果
 */
- (WCTRCPrivateContactUploadImageResponseResult *)privateContactUploadImage:(CPImage *)image
                                                                   fileName:(NSString *)fileName
                                                                   mineType:(NSString *)mineType
                                                                      error:(NSError **)error;


/**
 *  更新連絡人
 *
 *  @param contactUpdateInfoRequest 需求物件
 *  @param error                    回傳錯誤
 *
 *  @return 更新連絡人結果
 */
- (WCTRCPrivateContactUpdateResponseResult *)updatePrivateContactWithContactUpdateInfoRequest:(WCTRCContactUpdateInfoRequest *)contactUpdateInfoRequest
                                                                                        error:(NSError **)error;


/**
 *  手動新增連絡人
 *
 *  @param contactCreationWithImageRequest 手動新增連絡人需求物件
 *  @param error    回傳錯誤
 *
 *  @return  手動新增連絡人結果
 */
- (WCTRCPrivateContactManualUploadResponseResult *)manualUploadPrivateContactWithContactCreationWithImageRequest:(WCTRCContactCreationWithImageRequest *)contactCreationWithImageRequest
                                                                                                           error:(NSError **)error;


/**
 新增名片(協助掃描者專用API).
 
 *  @param contactCreationWithImageRequest 手動新增連絡人需求物件
 *  @param contactReceiverAccountGuid 要傳入指定要掃給哪一個帳號的GUID
 *  @param error    回傳錯誤
 *
 *  @return  手動新增連絡人結果 */
- (WCTRCPrivateContactScanUploadResponseResult *)scanUploadPrivateContactWithContactCreationWithImageRequest:(WCTRCContactCreationWithImageRequest *)contactCreationWithImageRequest
                                                                                  contactReceiverAccountGuid:(NSString *)contactReceiverAccountGuid
                                                                                                       error:(NSError **)error;


/**
 *  刪除連絡人
 *
 *  @param contactGuids 連絡人 guids
 *  @param error       回傳錯誤
 *
 *  @return 刪除連絡人結果
 */
- (WCTRCPrivateContactDeleteResponseResult *)deletePrivateContactWithContactGuids:(NSArray *)contactGuids
                                                                            error:(NSError **)error;


/**
 *  刪除連絡人
 *
 *  @param contactGuids 連絡人 guids
 *  @param progressHandler 回報處理進度
 *  @param error       回傳錯誤
 *
 *  @return 刪除連絡人結果
 */
- (WCTRCPrivateContactDeleteResponseResult *)deletePrivateContactWithContactGuids:(NSArray *)contactGuids
                                                                  progressHandler:(void(^)(CGFloat progress))progressHandler
                                                                            error:(NSError **)error;

/**
 *  取得類別下連絡人列表
 *
 *  @param contactGetByCategoryRequest 需求物件
 *  @param error    回傳錯誤
 *
 *  @return 取得類別下連絡人列表結果
 */
- (WCTRCPrivateContactListResponseResult *)privateContactListWithContactGetByCategoryRequest:(WCTRCContactGetByCategoryRequest *)contactGetByCategoryRequest
                                                                                       error:(NSError **)error;

/**
 *  取得帳號下連絡人列表
 *
 *  @param contactGetByAccountRequest 需求物件
 *  @param error    回傳錯誤
 *
 *  @return 取得類別下連絡人列表結果
 */
- (WCTRCPrivateContactListResponseResult *)privateContactListWithContactGetByAccountRequest:(WCTRCContactGetByAccountRequest *)contactGetByAccountRequest
                                                                                      error:(NSError **)error;

/**
 *  取得帳號下連絡人數量
 *
 *  @param accountGuid 連絡人id
 *  @param error    回傳錯誤
 *
 *  @return 取得帳號下連絡人數量結果
 */
- (WCTRCPrivateContactNumberResponseResult *)privateContactNumberUnderAccountWithAccountGuid:(NSString *)accountGuid
                                                                                       error:(NSError **)error;

/**
 *  連絡人新增or移除至我的最愛
 *
 *  @param contactsSetFavoriteRequest 需求物件
 *  @param error                      回傳錯誤
 *
 *  @return 連絡人新增or移除至我的最愛結果
 */
- (WCTRCPrivateContactToggleFavoriteResponseResult *)toggleFavoriteContactWithContactsSetFavoriteRequest:(WCTRCContactsSetFavoriteRequest *)contactsSetFavoriteRequest
                                                                                                   error:(NSError **)error;

/**
 *  連絡人設定成已校正
 *
 *  @param contactGuids 連絡人 guid of Array
 *  @param error        回傳錯誤
 *
 *  @return 連絡人設定成已校正結果
 */
- (WCTRCPrivateContactListSetCorrectedResponseResult *)privateContactListSetCorrectedWithContactGuids:(NSArray *)contactGuids
                                                                                                error:(NSError **)error;

/**
 *  連絡人移除類別
 *
 *  @param contactsRemoveFromCategoryRequest 需求物件
 *  @param error        回傳錯誤
 *
 *  @return 連絡人設定成已校正結果
 */
- (WCTRCContactsRemoveFromCategoryResponseResult *)privateContactsRemoveCategoryWithContactsRemoveFromCategoryRequest:(WCTRCContactsRemoveFromCategoryRequest *)contactsRemoveFromCategoryRequest
                                                                                                                error:(NSError **)error;

/**
 *  取得連絡人資訊
 *
 *  @param contactGuid 連絡人 Guid
 *  @param error        回傳錯誤
 *
 *  @return 取得連絡人資訊結果
 */
- (WCTRCPrivateContactResponseResult *)privateContactWithContactGuid:(NSString *)contactGuid
                                                               error:(NSError **)error;


/**
 *  取得連絡人 ImageBase64 資訊
 *
 *  @param contactGuid 連絡人 Guid
 *  @param imageType 圖片類型 WCTRC_ImageType_Front, WCTRC_ImageType_Rear, WCTRC_ImageType_Logo
 *  @param error 回傳錯誤
 *
 *  @return 取得連絡人 ImageBase64 資訊結果
 */
- (WCTRCPrivateContactImageBase64ResponseResult *)privateContactImageBase64WithContactGuid:(NSString *)contactGuid
                                                                                 imageType:(NSString *)imageType
                                                                                     error:(NSError **)error;


/**
 *  取得連絡人 Image 資訊
 *
 *  @param contactGuid 連絡人 Guid
 *  @param imageType 圖片類型 WCTRC_ImageType_Front, WCTRC_ImageType_Rear, WCTRC_ImageType_Logo
 *  @param error 回傳錯誤
 *
 *  @return 取得連絡人 Image 資訊結果
 */
- (CPImage *)privateContactImageWithContactGuid:(NSString *)contactGuid
                                      imageType:(NSString *)imageType
                                          error:(NSError **)error;

/**
 *  取得連絡人 Image 資訊 (NSData)
 *
 *  @param contactGuid 連絡人 Guid
 *  @param imageType 圖片類型 WCTRC_ImageType_Front, WCTRC_ImageType_Rear, WCTRC_ImageType_Logo
 *  @param error 回傳錯誤
 *
 *  @return NSData 取得連絡人 Image 資訊結果
 */
- (NSData *)privateContactImageDataWithContactGuid:(NSString *)contactGuid
                                         imageType:(NSString *)imageType
                                             error:(NSError **)error;




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instace Private Contact Method (V 1.3)

/**
 *  私有名片多層類別操作
 *
 *  @param contactSetCategoriesRequest 需求物件
 *  @param error                       回傳錯誤
 *
 *  @return 私有名片多層類別操作結果
 */
- (WCTRCPrivateContactOperateCategoryResponseResult *)privateContactOperateMultiLayerCategoryWithContactSetCategoriesRequest:(WCTRCContactSetCategoriesRequest *)contactSetCategoriesRequest
                                                                                                                       error:(NSError **)error;








////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Public Contact Method

/**
 *  公有連絡人下載
 *
 *  @param contactGuids 連絡人 Guids
 *  @param error        回傳錯誤
 *
 *  @return 公有連絡人下載結果
 */
- (WCTRCPublicContactsDownloadByContactGuidReponseResult *)publicContactsDownloadWithContactGuids:(NSArray *)contactGuids
                                                                                            error:(NSError **)error;


/**
 *  取得分享帳號
 *
 *  @param error 回傳錯誤
 *
 *  @return 取得分享帳號下載結果
 */
- (WCTRCSharedAccountsResponseResult *)shareAccountsWithError:(NSError **)error;


/**
 *  透過帳號取得連絡人數量
 *
 *  @param error 回傳錯誤
 *
 *  @return 取得透過帳號取得連絡人數量結果
 */
- (WCTRCContactNubmerByAccountGuidsResponseResult *)contactNumberWithAccountGuids:(NSArray *)accountGuids
                                                                            error:(NSError **)error;

/**
 *  取得所有連絡人數量
 *
 *  @param error 回傳錯誤
 *
 *  @return 取得透過帳號取得連絡人數量結果
 */
- (WCTRCAllContactNumberResponseResult *)allContactNumberWithError:(NSError **)error;


/**
 *  取得帳號下所有連絡人數量
 *
 *  @param error 回傳錯誤
 *
 *  @return 取得帳號下所有連絡人數量結果
 */
- (WCTRCContactNumberBySelfResponseResult *)contactNumberWithSelfByError:(NSError **)error;


/**
 *  取得連絡人狀態
 *
 *  @param contactGuids 連絡人 guids
 *  @param error        回傳錯誤
 *
 *  @return 取得連絡人狀態結果
 */
- (WCTRCSharedContactStatusResponseResult *)shareContactStatusWithContactGuids:(NSArray *)contactGuids
                                                                         error:(NSError **)error;

/**
 *  取得帳號下，連絡人們基本資訊
 *
 *  @param contactGetByAccountRequest 需求物件
 *  @param error        回傳錯誤
 *
 *  @return 取得帳號下，連絡人們基本資訊結果
 */
- (WCTRCContactListPageResponseResult *)shareContactListWithContactGetByAccountRequest:(WCTRCContactGetByAccountRequest *)contactGetByAccountRequest
                                                                             withError:(NSError **)error;


/**
 *  取得連絡人的分享狀態
 *
 *  @param contactGuid 連絡人 guid
 *  @param error       回傳錯誤
 *
 *  @return 取得類別下，取得連絡人的分享狀態結果
 */
- (WCTRCShareStatusForOneContactResponseResult *)shareStatusForOneContactWithContactGuid:(NSString *)contactGuid
                                                                                   error:(NSError **)error;
/**
 *  連絡人分享操作
 *
 *  @param contactShareAccountRequest 需求物件
 *  @param error                      回傳錯誤
 *
 *  @return 連絡人分享操作結果
 */
- (WCTRCContactShareAccountResponseResult *)contactShareOperationWithContactShareAccountRequest:(WCTRCContactShareAccountRequest *)contactShareAccountRequest
                                                                                          error:(NSError **)error;
/**
 *  連絡人分享操作
 *
 *  @param contactShareAccountRequest 需求物件
 *  @param progressHandler            回報處理進度
 *  @param error                      回傳錯誤
 *
 *  @return 連絡人分享操作結果
 */
- (WCTRCContactShareAccountResponseResult *)contactShareOperationWithContactShareAccountRequest:(WCTRCContactShareAccountRequest *)contactShareAccountRequest
                                                                                progressHandler:(void(^)(CGFloat progress))progressHandler
                                                                                          error:(NSError **)error;


/**
 *  設定名片擁有者
 *
 *  @param contactGuids        連絡人guids
 *  @param newOwnerAccountGuid 新的擁有者
 *  @param error               回傳錯誤
 *
 *  @return 設定名片擁有者結果
 */
- (WCTRCSetOwnerOfContactResponseResult *)setContactOwnerWithContactGuids:(NSArray *)contactGuids
                                                      newOwnerAccountGuid:(NSString *)newOwnerAccountGuid
                                                                    error:(NSError **)error;

/**
 *  設定名片擁有者
 *
 *  @param contactGuids        連絡人guids
 *  @param newOwnerAccountGuid 新的擁有者
 *  @param progressHandler            回報處理進度
 *  @param error               回傳錯誤
 *
 *  @return 設定名片擁有者結果
 */
- (WCTRCSetOwnerOfContactResponseResult *)setContactOwnerWithContactGuids:(NSArray *)contactGuids
                                                      newOwnerAccountGuid:(NSString *)newOwnerAccountGuid
                                                          progressHandler:(void(^)(CGFloat progress))progressHandler
                                                                    error:(NSError **)error;

/**
 *  匯出指定的contact到CRM
 *
 *  @param request        名片 request
 *  @param error               回傳錯誤
 *
 *  @return 設定名片擁有者結果
 */
- (WCTRCContactListPageResponseResult *)contactListWithRequest:(WCTRCContactGetByGuidListRequest *)request
                                                         error:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Public Contact Method (V 1.3)

/**
 取得公有區裡,某帳號所擁有並且有分享給我,但卻不在他的私有區裡的名片數量.
 
 @param accountGuid 目標帳號
 @param error 回傳錯誤
 @return 名片數量結果
 */
- (WCTRCSharedContactNumberIsntAccountCategoryResponseResult *)sharedContactNumberIsntAccountCategoryWithAccountGuid:(NSString *)accountGuid
                                                                                                               error:(NSError **)error;



/**
 傳入公有區裡的類別guid列表,取回這些類別裡的名片數量
 
 @param categoryGuids 類別 ID's
 @param categoryOwnerGuid 類別擁有者
 @param error 回傳錯誤
 @return 名片數量結果
 */
- (WCTRCMLCategorySharedContactNumberWithCategoryGuidsResponseResult *)sharedContactNumberWithCategoryGuids:(NSArray *)categoryGuids
                                                                                          categoryOwnerGuid:(NSString *)categoryOwnerGuid
                                                                                                      error:(NSError **)error;


/**
 取得公有區裡,某帳號所擁有並且有分享給我,但卻不在他的私有區裡的名片列表
 
 
 @param request 名片列表需求物件
 @param error 回傳錯誤
 @return 名片列表
 */
- (WCTRCContactListPageResponseResult *)sharedContactListIsntAccountCategoryWithRequest:(WCTRCContactGetByAccountRequest *)request
                                                                                  error:(NSError **)error;



/**
 取得公有區裡,某一個類別裡的名片列表
 
 @param request 名片列表需求物件
 @param error 回傳錯誤
 @return 名片列表
 */
- (WCTRCContactListPageResponseResult *)sharedContactListWithRequest:(WCTRCContactGetByCategoryAndOwnerVoRequest *)request
                                                               error:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instace Multi Layer Category Method


/**
 建立多層多類別
 
 @param multiLayerCategoryCreationRequest 建立物件
 @param error 回傳錯誤
 @return 建立多層多類別結果
 */
- (WCTRCMLCategoryCreationResponseResult *)createMultiLayerCategoryWithCreationRequest:(WCTRCMLCategoryCreationRequest *)multiLayerCategoryCreationRequest
                                                                                 error:(NSError **)error;


/**
 建立多層多類別路徑
 
 @param categoryPathCreationRequest 建立物件
 @param error 回傳錯誤
 @return 建立多層多類別結果
 */
- (WCTRCMLCategoryPathCreationResponseResult *)createMultiLayerCategoryPathWithCreationRequest:(WCTRCMLCategoryPathCreationRequest *)categoryPathCreationRequest
                                                                                         error:(NSError **)error;


/**
 刪除多層多類別
 
 @param categoryGuid 類別 ID
 @param error 回傳錯誤
 @return 刪除多層多類別結果
 */
- (WCTRCMLCategoryDeleteResponseResult *)deleteMultiLayerCategoryWithCategoryGuid:(NSString *)categoryGuid
                                                                            error:(NSError **)error;


/**
 *  取得指定帳號所擁有的全部多層類別
 *
 *  @param error 回傳錯誤
 *
 *  @return 帳號類別列表
 */
- (WCTRCMLCategoryInfoListResponseResult *)multiLayerCategoryListWithAccountGuid:(NSString *)accountGuid
                                                                           error:(NSError **)error;



/**
 取回指定帳號底下,可做為類別移動目的地的類別列表
 
 @param movedCategoryGuid 要被移動的類別
 @param error 回傳錯誤
 @return 類別列表
 */
- (WCTRCMLCategoryInfoListResponseResult *)movableMultiLayerCategoryListWithAccountGuid:(NSString *)accountGuid movedCategoryGuid:(NSString *)movedCategoryGuid
                                                                                  error:(NSError **)error;



/**
 取得指定帳號,指定類別底下一層的類別列表,如果不指定類別,會取得該帳號的第一層類別
 
 @param parentCategoryGuid 父類別 ID
 @param needMarkHasChild 標示是否有子類別
 @param error 回傳錯誤
 @return 類別列表
 */
- (WCTRCMLCategoryInfoListResponseResult *)multiLayerCategoryOneLayerListWithParentCategoryGuid:(NSString *)parentCategoryGuid
                                                                               needMarkHasChild:(BOOL)needMarkHasChild
                                                                                          error:(NSError **)error;



/**
 按使用者設定的顯示順序,取回登入帳號底下,被標為釘選的類別列表
 
 @param error 回傳錯誤
 @return 釘選的類別列表
 */
- (WCTRCMLCategoryInfoListResponseResult *)toggledMultiLayerCategoryListWithError:(NSError **)error;



/**
 設定釘選類別順序
 
 @param request 類別順序 物件
 @param error 回傳錯誤
 @return 設定釘選類別順序結果
 */
- (WCTRCMLCategorySetOrderForToggledCategoriesResponseResult *)setOrderToggledMultiLayerCategoryListWithRequest:(WCTRCMLSetOrderListRequest *)request
                                                                                                          error:(NSError **)error;


/**
 將多個類別指定或取消為登入帳號的釘選類別
 
 @param categoryToggleUsualVo 指定或取消釘選類別物件
 @param error 回傳錯誤
 @return 指定或取消釘選類別結果
 */
- (WCTRCMLCategoryToggledCategoriesResponseResult *)setToggledMultiLayerWithCategoryToggleUsualVo:(WCTRCCategoryToggleUsualVo *)categoryToggleUsualVo
                                                                                            error:(NSError **)error;


/**
 修改指定類別的名稱
 
 @param guid 類別 guid
 @param categoryName 類別名稱
 @param error 回傳錯誤
 @return 修改指定類別的名稱結果
 */
- (WCTRCMLCategoryUpdateResponseResult *)updateMultiLayerCategoryWithGuid:(NSString *)guid
                                                             categoryName:(NSString *)categoryName
                                                                    error:(NSError **)error;


/**
 將類別移動到某一個指定的父類別底下
 
 @param guid 類別 Guid
 @param parentCategoryGuid 父類別id
 @param categoryOwnerGuid 類別擁有者id
 @param error 錯誤回傳
 @return  類別移動到某一個指定的父類別底下結果
 */
- (WCTRCMLCategoryMoveResponseResult *)moveMultiLayerCategoryWithGuid:(NSString *)guid
                                                 toParentCategoryGuid:(NSString *)parentCategoryGuid
                                                    categoryOwnerGuid:(NSString *)categoryOwnerGuid
                                                                error:(NSError **)error;



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Search Method

/**
 *  透過關鍵字搜尋公司連絡人
 *
 *  @param keyword 關鍵字
 *  @param error   回傳錯誤
 *
 *  @return 關鍵字搜尋連絡人結果
 */
- (WCTRCSearchPublicContactsByKeywordResponseResult *)searchPublicContactsByKeywordWithKeyword:(NSString *)keyword
                                                                                         error:(NSError **)error;


/**
 *  透過關鍵字搜尋回傳有排序過的連絡人
 *
 *  @param keyword                    關鍵字
 *  @param page                       頁碼
 *  @param pageSize                   每頁幾筆
 *  @param contactsimpleinfoSortField 依連絡人欄位排序
 *  @param sortDirection              排序冪次
 *  @param error                      回傳錯誤
 *
 *  @return 關鍵字搜尋回傳有排序過的連絡人結果
 */
- (WCTRCSearchPublicContactsByKeywordSortedResponseResult *)searchPublicContactsByKeywordSortedWithKeyword:(NSString *)keyword
                                                                                                      page:(NSUInteger)page
                                                                                                  pageSize:(NSUInteger)pageSize
                                                                                contactsimpleinfoSortField:(NSString *)contactsimpleinfoSortField
                                                                                             sortDirection:(NSString *)sortDirection
                                                                                                     error:(NSError **)error;

/**
 *  透過關鍵字搜尋我的連絡人
 *
 *  @param keyword 關鍵字
 *  @param error   回傳錯誤
 *
 *  @return 關鍵字搜尋任務中的連絡人結果
 */
- (WCTRCSearchPrivateContactsByKeywordResponseResult *)searchPrivateContactsByKeywordWithKeyword:(NSString *)keyword
                                                                                           error:(NSError **)error;

/**
 *  透過條件組合搜尋聯絡人，WCTRCAdvancedSearchVoRequest中的contactRangeType決定是搜尋我的聯絡人或是共享聯絡人
 *
 *  @param request 搜尋條件
 *  @param error   回傳錯誤
 *
 *  @return nil, 代表需求沒透過網路傳送出去
 */
- (WCTRCAdvancedSearchContactListResposeResult *)advancedSearchWithRequest:(WCTRCAdvancedSearchVoRequest *)request
                                                                     error:(NSError **)error;




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Notice methods

/**
 * 取得此帳號還沒讀過的通知筆數
 *  @param error         回傳錯誤
 *
 * @return WCTRCLongResponseResult value of unread notice
 */
- (WCTRCMessageResponseResult *)setLastNoticeReadWithError:(NSError **)error;


/**
 * 取得此帳號還沒讀過的通知筆數
 *
 * @param noticeCategory WCTRC_NoticeCategory_All, WCTRC_NoticeCategory_Contact,
 *                       WCTRC_NoticeCategory_Activity, WCTRC_NoticeCategory_Task,
 *                       WCTRC_NoticeCategory_System
 *  @param error         回傳錯誤
 *
 * @return WCTRCLongResponseResult value of unread notice
 */
- (WCTRCLongResponseResult *)unreadNoticeWithNoticeCategory:(NSString *)noticeCategory
                                                      error:(NSError **)error;


/**
 * 取得此帳號的所有通知
 *
 * @param noticeCategory WCTRC_NoticeCategory_All, WCTRC_NoticeCategory_Contact,
 *                       WCTRC_NoticeCategory_Activity, WCTRC_NoticeCategory_Task,
 *                       WCTRC_NoticeCategory_System
 *  @param error         回傳錯誤
 *
 * @return WCTNoticeListResponseResult all notice list
 */
- (WCTRCNoticeListResponseResult *)noticeListWithNoticeCategory:(NSString *)noticeCategory
                                                          error:(NSError **)error;


/**
 * 取得此帳號的所有通知
 *
 * @param noticeCategory WCTRC_NoticeCategory_All, WCTRC_NoticeCategory_Contact,
 *                       WCTRC_NoticeCategory_Activity, WCTRC_NoticeCategory_Task,
 *                       WCTRC_NoticeCategory_System
 *  @param noticeReadStatus 通知的狀態 WCTRC_NoticeReadStatus
 *  @param pageIndex 通知的頁次
 *  @param pageSize 一次取得幾個通知
 *  @param sortField 排序欄位 WCTRC_NoticeSortField_Type，WCTRC_NoticeSortField_Category，WCTRC_NoticeSortField_Time
 *  @param sortDirection 排序方向 WCTRC_NoticeSortDirection_ASC，WCTRC_NoticeSortDirection_DESC
 *  @param error         回傳錯誤
 *
 * @return WCTNoticeListPageResponseResult notice of a page
 */
- (WCTRCNoticeListPageResponseResult *)noticeListPageWithNoticeCategory:(NSString *)noticeCategory
                                                       noticeReadStatus:(NSString *)noticeReadStatus
                                                              pageIndex:(NSInteger)pageIndex
                                                               pageSize:(NSInteger)pageSize
                                                              sortField:(NSString *)sortField
                                                          sortDirection:(NSString *)sortDirection
                                                                  error:(NSError **)error;

/**
 * 取得通知內文
 *
 * @param noticeGuid 要取得的通知guid
 *  @param error         回傳錯誤
 *
 * @return WCTRCLongResponseResult value of unread notice
 */
- (WCTRCMessageResponseResult *)noticeContentWithGuid:(NSString *)noticeGuid
                                                error:(NSError **)error;




/**
 發送分享聯絡人至使用者帳號的通知
 
 @param contacts The array of Contact's guid
 @param accounts The array of Account's guid
 @param error 回傳錯誤
 @return WCTRCMessageResponseResult Result
 */
- (WCTRCMessageResponseResult *)noticeSendShareContacts:(NSArray *)contacts
                                      toReceiveAccounts:(NSArray *)accounts
                                                  error:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance NormalUserSetting Method

/**
 *  取得所有設定值
 *
 *  @param error 回傳錯誤
 *
 *  @return 所有設定
 */
- (WCTRCAllUserSettingsResponseResult *)allUserSettingsResponseWithError:(NSError **)error;


/**
 *  支援地圖類型
 *
 *  @param error 回傳錯誤
 *
 *  @return 地圖類型結果
 */
- (WCTRCSupportMapEngineResponseResult *)supportMapEngineResponseWithError:(NSError **)error;


/**
 *  取得使用地圖
 *
 *  @param error 回傳錯誤
 *
 *  @return 使用地圖
 */
- (WCTRCNowMapEngineResponseResult *)nowMapEngineResponseWithError:(NSError **)error;


/**
 *  東方姓名順序
 *
 *  @param error 回傳錯誤
 *
 *  @return 東方姓名順序
 */
- (WCTRCAsianSurnameOrderResult *)asianSurnameOrderWithError:(NSError **)error;



/**
 *  西方姓名順序
 *
 *  @param error 回傳錯誤
 *
 *  @return 西方姓名順序
 */
- (WCTRCWesternSurnameOrderSettingResult *)westernSurnameOrderWithError:(NSError **)error;


/**
 *  是否自動分享
 *
 *  @param error 回傳錯誤
 *
 *  @return 自動分享
 */
- (WCTRCAutoShareResponseResult *)autoShareSettingWithError:(NSError **)error;


/**
 *  取得目前分享列表
 *
 *  @param shareItem 要取得的列表是哪一種分享資料的列表，參考 WCTRestObject+ParameterDefine.h WCTRC_ShareItem
 *  @param error 回傳錯誤
 *
 *  @return 目前分享列表
 */
- (WCTRCAccountSharedTargetsResponseResult *)accountSharedTargetsWithShareItem:(NSString *)shareItem
                                                                         error:(NSError **)error;


/**
 *  是否自動分享給全公司聯絡人
 *
 *  @param error 回傳錯誤
 *
 *  @return 全公司共享 [ALL, BY_SETTING]
 */
- (WCTRCShareRoleResponseResult *)globalShareRoleWithError:(NSError **)error;


/**
 *  中文辨識結果輸出設定
 *
 *  @param error 回傳錯誤
 *
 *  @return 參考WCTRC_ChineseRecognizeOutput
 */
- (WCTRCChineseRecognizeResultOutputResponseResult*)chineseRecognizeResultOutputWithError:(NSError **)error;


/**
 * 依欄位排序設定
 *
 *  @param error 回傳錯誤
 *
 *  @return 參考WCTRC_SortByField
 */
- (WCTRCSortByFieldResponseResult*)sortByFieldWithError:(NSError **)error;



/**
 * 取得所有自訂欄位
 *
 *  @param category 要取得的自訂欄位類別 (WCTRC_CustomField_Category_Contact，WCTRC_CustomField_Category_Task，WCTRC_CustomField_Category_Activity)
 *  @param error 回傳錯誤
 *
 *  @return 參考WCTRC_SortByField
 */
- (WCTRCCustomFieldListResponseResult *)customFieldsWithCategory:(NSString *)category
                                                           error:(NSError **)error;

/**
 * 取得最大可同步聯絡人數量
 *
 *  @param error 回傳錯誤
 *
 *  @return 可同步數量
 */
- (NSInteger)maxPrivateContactsCountWithError:(NSError **)error;


/**
 * 取得公有區最大聯絡人數量
 *
 *  @param error 回傳錯誤
 *
 *  @return 可同步數量
 */
- (NSInteger)maxPublicContactsCountWithError:(NSError **)error;

/**
 * 取得Server最大聯絡人總數量
 *
 *  @param error 回傳錯誤
 *
 *  @return Server最大聯絡人總數量
 */
- (NSInteger)maxServerContactsCountWithError:(NSError **)error;


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - setting methods


/**
 *  設定東方姓名順序
 *
 *  @param nameOrder WCTRC_NameOrder_FirstNameLastName or WCTRC_NameOrder_LastNameFirstName
 *  @param error     回傳錯誤
 *
 *  @return 東方姓名順序結果
 */
- (WCTRCSetSurnameOrderResponseResult *)setAsianSurnameOrderWithNameOrder:(NSString *)nameOrder
                                                                    error:(NSError **)error;

/**
 *  設定西方姓名順序
 *
 *  @param nameOrder WCTRC_NameOrder_FirstNameLastName or WCTRC_NameOrder_LastNameFirstName
 *  @param error     回傳錯誤
 *
 *  @return 西方姓名順序結果
 */
- (WCTRCSetSurnameOrderResponseResult *)setWesternSurnameOrderWithNameOrder:(NSString *)nameOrder
                                                                      error:(NSError **)error;

/**
 *  設定地圖類型
 *
 *  @param mapEngine WCTRC_MapEngine_Auto, WCTRC_MapEngine_Google, WCTRC_MapEngine_Baidu
 *  @param error     回傳錯誤
 *
 *  @return 地圖類型結果
 */
- (WCTRCSetUserMapEngineResponseResult *)setMapEngineWithMapEngine:(NSString *)mapEngine
                                                             error:(NSError **)error;



/**
 *  設定是否自動分享
 *
 *  @param enable YES, or NO
 *  @param error     回傳錯誤
 *
 *  @return 結果
 */
- (WCTRCSetAutoShareSettingResponseResult *)setAutoShareSettingWithEnable:(BOOL)enable
                                                                    error:(NSError **)error;


/**
 *  新增自動分享的對象
 *
 *  @param request 要新增的對象
 *  @param error     回傳錯誤
 *
 *  @return 結果
 */
- (WCTRCMessageResponseResult *)addAccountSharedTargetWithRequest:(WCTRCAccountSharedTargetRequest *)request
                                                            error:(NSError **)error;


/**
 *  移除自動分享的對象
 *
 *  @param request 要移除的對象
 *  @param error     回傳錯誤
 *
 *  @return 結果
 */
- (WCTRCMessageResponseResult *)removeAccountSharedTargetWithRequest:(WCTRCAccountSharedTargetRequest *)request
                                                               error:(NSError **)error;


/**
 *  更新自動分享的對象列表
 *
 *  @param request  更新的內容
 *  @param error     回傳錯誤
 *
 *  @return 結果
 */
- (WCTRCMessageResponseResult *)updateAccountSharedTargetListWithRequest:(WCTRCAccountSharedTargetListRequest *)request
                                                                   error:(NSError **)error;


/**
 *  設定中文辨識結果輸出
 *
 *  @param output  WCTRC_ChineseRecognizeOutput_ContactContent, WCTRC_ChineseRecognizeOutput_CHT, WCTRC_ChineseRecognizeOutput_CHS
 *  @param error     回傳錯誤
 *
 *  @return 西方姓名順序結果
 */
- (WCTRCSetChineseRecognizeOutputResponseResult *)setChineseRecognizeWithOutput:(NSString *)output
                                                                          error:(NSError **)error;


/**
 *  設定依欄位排序
 *
 *  @param field  WCTRC_SortByField_Name, WCTRC_SortByField_Company, WCTRC_SortByField_CreateTime, WCTRC_SortByField_ModifiedTime
 *  @param error     回傳錯誤
 *
 *  @return 西方姓名順序結果
 */
- (WCTRCSetSortByFieldResponseResult *)setSortByFieldWithField:(NSString *)field
                                                         error:(NSError **)error;


/**
 *  設定辨識後是否進入編輯
 *
 *  @param enable YES, or NO
 *  @param error     回傳錯誤
 *
 *  @return 結果
 */
- (WCTRCSetEditAfterRecogResponseResult *)setEditAfterRecogSettingWithEnable:(BOOL)enable
                                                                       error:(NSError **)error;


/**
 *  設定辨識後是否進入備註
 *
 *  @param enable YES, or NO
 *  @param error     回傳錯誤
 *
 *  @return 結果
 */
- (WCTRCSetAddNoteInfoAfterRecogResponseResult *)setAddNoteInfoAfterRecogWithEnable:(BOOL)enable
                                                                              error:(NSError **)error;


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - CRM settings methods



/**
 *  設定CRM
 *
 *  @param settings  settings of CRM
 *  @param error     回傳錯誤
 *
 *  @return 設定結果
 */
- (WCTRCMessageResponseResult *)setCRMSettingsWithSettings:(WCTRCCRMSettings *)settings
                                                     error:(NSError **)error;


/**
 *  取得CRM設定
 *
 *  @param error     回傳錯誤
 *
 *  @return 設定結果
 */
- (WCTRCCRMSettingsResponseResult *)CRMSettingsWithError:(NSError **)error;


/**
 *  取得系統CRM種類
 *
 *  @param error     回傳錯誤
 *
 *  @return 設定結果
 */
- (WCTRCCRMSystemResponseResult *)CRMSystemWithError:(NSError **)error;


/**
 *  登出CRM
 *
 *  @param CRMType   要登出的CRM類型
 *  @param error     回傳錯誤
 *
 *  @return 設定結果
 */
- (WCTRCMessageResponseResult *)logoutCRMWithType:(NSString *)CRMType
                                            error:(NSError **)error;


/**
 *  取得系統CRM種類
 *
 *  @param error     回傳錯誤
 *
 *  @return 設定結果
 */
- (WCTRCUserExportInfoResponseResult *)lastCRMExportInfoWithError:(NSError **)error;


/**
 *  設定CRM匯出模式
 *
 *  @param CRMType 要設定的CRMType
 *  @param CRMExportMode 匯出模式
 *  @param CRMCompanyAssignmentOption 公司匯出指定
 *  @param error 回傳錯誤
 *
 *  @return nil, 代表需求沒透過網路傳送出去
 */

- (WCTRCMessageResponseResult *)setCRMExportModeWithCRMType:(NSString *)CRMType
                                              CTMExportMode:(NSString *)CRMExportMode
                                 CRMCompanyAssignmentOption:(NSString *)CRMCompanyAssignmentOption
                                                      error:(NSError **)error;


/**
 *  設定CRM - 以access token方式使用
 *
 *  @param tokenRequest 要設定的CRM token 資料
 *  @param error 回傳錯誤
 *  @return nil, 代表需求沒透過網路傳送出去
 */
- (WCTRCMessageResponseResult *)setCRMTokenWithTokenRequest:(WCTRCUpdateCRMTokenRequest *)tokenRequest
                                                      error:(NSError **)error;

/**
 *  透過Server取得salesforce的OAuth token
 *
 *  @param code salesforce的approval code
 *  @param error 回傳錯誤
 *
 *  @return nil, 代表需求沒透過網路傳送出去
 */
- (WCTRCCRMUserAgentOAuthTokenResponseResult *)salesforceOAuthTokenWithCode:(NSString *)code
                                                                      error:(NSError **)error;




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - contact server settings methods

/**
 *  是否支援Exchange server功能，server version 6.1以上才支援
 *
 *  @return 是否支援Exchange server功能
 */
- (BOOL)isExchangeServerSupported;

/**
 *  設定個人exchange server
 *
 *  @param settings  settings of CRM
 *  @param error     回傳錯誤
 *
 *  @return 設定結果
 */
- (WCTRCMessageResponseResult *)setExchangeServerSettingsWithSettings:(WCTRCExchangeServerSettings *)settings
                                                                error:(NSError **)error;



/**
 *  取得個人exchange server設定
 *
 *  @param error     回傳錯誤
 *
 *  @return 設定結果
 */
- (WCTRCExchangeServerSettingsResponseResult *)exchangeServerSettingsWithError:(NSError **)error;


/**
 *  取得系統exchange server類型與url
 *
 *  @param error     回傳錯誤
 *
 *  @return 設定結果
 */
- (WCTRCExchangeServerSystemResponseResult *)exchangeServerInfoWithError:(NSError **)error;


/**
 *  登出Exchange server
 *
 *  @param error     回傳錯誤
 *
 *  @return 設定結果
 */
- (WCTRCMessageResponseResult *)logoutExchangeServerWithError:(NSError **)error;


/**
 *  取得系統Exchange server最後同步資訊
 *
 *  @param error     回傳錯誤
 *
 *  @return 設定結果
 */
- (WCTRCUserExportInfoResponseResult *)lastExchangeServerExportInfoWithError:(NSError **)error;


/**
 *  設定Exchange server匯出模式
 *
 *  @param mode 匯出模式
 *  @param error 回傳錯誤
 *
 *  @return nil, 代表需求沒透過網路傳送出去
 */

- (WCTRCMessageResponseResult *)setExchangeServerExportModeWithMode:(NSString *)mode
                                                              error:(NSError **)error;

/**
 *  設定Exchange server token
 *
 *  @param tokenSettings token info
 *  @param error 回傳錯誤
 *
 *  @return nil, 代表需求沒透過網路傳送出去
 */
- (WCTRCMessageResponseResult *)setExchangeServerTokenWithTokenRequest:(WCTRCExchangeServerTokenSettings *)tokenSettings
                                                                 error:(NSError **)error;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Statistic Method

/**
 *  取得活動資訊
 *
 *  @param error 回傳錯誤
 *
 *  @return 活動資訊
 */
- (WCTRCTotalActivityInfosResponseResult *)totalActivityInfosWithError:(NSError **)error;


/**
 *  取得授權資訊
 *
 *  @param error 回傳錯誤
 *
 *  @return　授權資訊
 */
- (WCTRCAuthorizedInfoResponseResult *)authorizedInfoWithError:(NSError **)error;


/**
 *  取得帳號下的聯絡人數量
 *
 *  @param error 回傳錯誤
 *
 *  @return 聯絡人數量
 */
- (WCTRCTotalContactInfosResponseResult *)totalContactInfosWithError:(NSError **)error;


/**
 *  取得儲存空間資訊
 *
 *  @param error 回傳錯誤
 *
 *  @return 儲存空間資訊
 */
- (WCTRCStorageInfoResponseResult *)storageInfoWithError:(NSError **)error;


/**
 *  取得所有任務資訊
 *
 *  @param error 回傳錯誤
 *
 *  @return 任務資訊
 */
- (WCTRCTotalTaskInfosResponseResult *)totalTaskInfosWithError:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Text Method

- (TestResponseObject *)textResponseObjectByGETMethodWithError:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Sync Method

/**
 * 同步動作，在server端建立群組
 * @param categoryRequest 要建立的群組資訊
 * @param error 回傳錯誤
 * @return WCTRCSyncCategoryCreateResponseResult 建立群組後server的回應
 */
- (WCTRCSyncCategoryCreateResponseResult *)syncCreateCategoryWithCategoryRequest:(WCTRCCategoryInfo *)categoryRequest
                                                                           error:(NSError **)error DEPRECATED_MSG_ATTRIBUTE("WCT ServerVersion 4.0.0");


/**
 * 同步動作，在server端刪除群組
 * @param categoryGuid 要刪除的群組GUID
 * @param error 回傳錯誤
 * @return WCTRCSyncCategoryDeleteResponseResult 刪除群組後server的回應
 */
- (WCTRCSyncCategoryDeleteResponseResult *)syncDeleteCategoryWithCategoryGuid:(NSString *)categoryGuid
                                                                        error:(NSError **)error DEPRECATED_MSG_ATTRIBUTE("WCT ServerVersion 4.0.0");


/**
 * 同步動作，在server端取得群組完整資料
 * @param categoryGuid 要取得的群組的guid
 * @param error 回傳錯誤
 * @return WCTRCDetailCategoryInfoResponseResult 取得群組後server的回應
 */
- (WCTRCDetailCategoryInfoResponseResult *)syncGetCategoryWithCategoryGuid:(NSString *)categoryGuid
                                                                     error:(NSError **)error DEPRECATED_MSG_ATTRIBUTE("WCT ServerVersion 4.0.0");


/**
 * 同步動作，在server端更新群組
 * @param categoryRequest 要更新的群組資訊
 * @param error 回傳錯誤
 * @return WCTRCSyncCategoryUpdateResponseResult 更新群組後server的回應
 */
- (WCTRCSyncCategoryUpdateResponseResult *)syncUpdateCategoryWithCategoryRequest:(WCTRCCategoryInfo *)categoryRequest
                                                                           error:(NSError **)error DEPRECATED_MSG_ATTRIBUTE("WCT ServerVersion 4.0.0");


/**
 * 同步動作，在server端建立聯絡人
 * @param request 要建立的群組資訊
 * @param error 回傳錯誤
 * @return WCTRCContactProfileResponseResult 建立聯絡人後server的回應
 */
- (WCTRCContactProfileResponseResult *)syncCreateContactWithRequest:(WCTRCContactCreationWithImageRequest *)request
                                                              error:(NSError **)error;


/**
 同步動作，檢查聯絡人是否存在
 
 @param contactGuid 連絡人ID
 @param accountGuid 帳號ID
 @param error 回傳錯誤
 @return 檢查聯絡人是否存在的回應
 */
- (WCTRCBooleanResponseResult *)syncIsExistContactWithContactGuid:(NSString *)contactGuid
                                                      accountGuid:(NSString *)accountGuid
                                                            error:(NSError **)error;


/**
 * 同步動作，在server端刪除聯絡人
 * @param contactGuid 要刪除的聯絡人的guid
 * @param accountGuid 要刪除聯聯絡人account guid
 * @param error 回傳錯誤
 * @return WCTRCSyncContactDeleteResponseResult 刪除聯絡人後server的回應
 */
- (WCTRCSyncContactDeleteResponseResult *)syncDeleteContactWithContactGuid:(NSString *)contactGuid
                                                               accountGuid:(NSString *)accountGuid
                                                                     error:(NSError **)error;


/**
 * 同步動作，在server端取得聯絡人完整資料
 * @param contactGuid 要取得的聯絡人的guid
 * @param error 回傳錯誤
 * @return WCTRCSyncContactInfoResponseResult 取得聯絡人後server的回應
 */
- (WCTRCDetailContactInfoResponseResult *)syncGetContactWithContactGuid:(NSString *)contactGuid
                                                                  error:(NSError **)error;


/**
 * 同步動作，在server端更新聯絡人
 * @param request 要更新的聯絡人的資訊
 * @param error 回傳錯誤
 * @return WCTRCContactProfileResponseResult 更新聯絡人後server的回應
 */
- (WCTRCContactProfileResponseResult *)syncUpdateContactWithRequest:(WCTRCContactUpdateInfoRequest *)request
                                                              error:(NSError **)error;

/**
 * 同步動作，取得需要更新的聯絡人項目
 * @param request 要更新的聯絡人的資訊
 * @param error 回傳錯誤
 * @return WCTRCContactProfileResponseResult 更新聯絡人後server的回應
 */
- (WCTRCSyncContactUpdateCheckResponseResult *)syncContactUpdateCheckWithRequest:(WCTRCContactUpdateCheckRequest *)request
                                                                           error:(NSError **)error;


/**
 * 同步動作，收到|UNCHECK|時要call這個api取得真正的動作
 * @param request 要更新的聯絡人的資訊
 * @param error 回傳錯誤
 * @return WCTRCContactProfileResponseResult 更新聯絡人後server的回應
 */
- (WCTRCSyncUpdateActionResponseResult *)syncContactCheckStatusWithRequest:(WCTRCSyncContactCheckRequest *)request
                                                                     error:(NSError **)error;


/**
 * 同步動作，是否需要重新比對(與上次同步時間超過30天)
 * @param contactLastRecTime syncInfo的contactLastRecTime
 * @param error 回傳錯誤
 * @return BOOL server回傳是否需要重新比對
 */
- (WCTRCSyncNeedForceCompareResponseResult *)syncNeedForceCompareWithContactLastRecTime:(NSDate *)contactLastRecTime
                                                                                  error:(NSError **)error;

/**
 * 同步動作，開始同步，傳送同步資訊給server, 並取得同步動作
 * @param syncInfo 要更新的聯絡人的資訊
 * @param error 回傳錯誤
 * @return WCTRCStartSyncResponseResult server回傳的同步動作列表
 */
- (WCTRCStartSyncResponseResult *)syncStartWithSyncInfo:(WCTRCSyncInfo *)syncInfo
                                                  error:(NSError **)error;

/**
 * 同步動作，同步完成
 * @param error 回傳錯誤
 * @return WCTRCMessageResponseResult 通知同步完成後server的回應
 */
- (WCTRCMessageResponseResult *)syncDoneWithError:(NSError **)error;






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance System Method

/**
 *  檢查伺服器是否運作中
 *
 *  @param error 回傳錯誤
 *  @return 檢查伺服器是否運作中的結果
 */
- (WCTRCServerIsAliveResponseResult *)serverIsAliveWithError:(NSError **)error;



/**
 取得伺服器版本
 
 @param error 回傳錯誤
 
 @return 伺服器版本結果
 */
- (WCTRCVersionInfoResponseResult *)serverVersionWithError:(NSError **)error;



/**
 檢查伺服器開放port
 
 @param error 回傳錯誤
 @return 伺服器port結果
 */
- (WCTRCServiceStatusResponseResult *)serverServiceStatusWithError:(NSError **)error;


/**
 伺服器是否註冊
 
 @param error 回傳錯誤
 @return YES, 已註冊
 */
- (WCTRCServerIsActivedResult *)serverIsActivedWithError:(NSError **)error;


/**
 伺服器是否要註冊
 
 @param error 回傳錯誤
 @return YES, 需要註冊
 */
- (WCTRCServerNeedBeActivedResult *)serverNeedBeActivedWithError:(NSError **)error;


/**
 伺服器URL資訊
 @param error 回傳錯誤
 @return dns
 */
- (WCTRCDNSNameResponseResult *)serverDNSNameWithError:(NSError **)error;


/**
 取得伺服器註冊狀態
 
 @param error 回傳錯誤
 @return 伺服器註冊狀態
 */
- (WCTRCServerRegisterStatusResponseResult *)serverRegisterStatusWithError:(NSError **)error __deprecated_msg("serverRegisterStatusWithError is deprecated. Use serverLicenseWithError.");;


/**
 取得server授權模式
 
 @param error 回傳錯誤
 @return 伺服器授權模式
 */
- (WCTRCServerLicenseResponseResult *)serverLicenseWithError:(NSError **)error;

/**
 取得server 資料庫建立時間
 
 @param error 回傳錯誤
 @return 伺服器資料庫建立時間
 */
- (WCTRCDateTimeResponseResult *)serverDBBuildTimeWithError:(NSError **)error;


/**
 取得資料庫是否包含多個公司, check完server version後才可以call
 
 @param error 回傳錯誤
 @return YES: 包含多公司，else NO
 */
- (BOOL)isMultiCompanyWithError:(NSError **)error;


/**
 取得共享聯絡人列表是否要顯示名片數, check完server version後才可以call
 
 @param error 回傳錯誤
 @return YES: get_accounts_share_to_me回傳的sharedContactCount都會是-1, 不顯示名片數;else NO
 */
- (BOOL)isSimplifyUIWithError:(NSError **)error;


/**
 取得顯示設定
 
 @param error 回傳錯誤
 @return WCTRCUIDisplayOptionResponseResult 回傳顯示設定
 */
- (WCTRCUIDisplayOptionResponseResult *)displayOptionWithError:(NSError **)error;



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Register Method


/**
 取得伺服器所有註冊資訊
 
 @param error 回傳錯誤
 @return 伺服器所有註冊資訊
 */
- (WCTRCAllRegistryListResponseResult *)serverAllRegistryWithError:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Subscribe Method

/**
 取得訂閱資訊
 
 @param error 回傳錯誤
 */
- (WCTRCSubscriptionInfoVoResponseResult *)subscriptionInfoVoResponseResultWithError:(NSError **)error;


/**
 更新訂閱資訊
 
 @param error 回傳錯誤
 */
- (WCTRCRefreshSubscriptionInfoToServerResult *)refreshSubscriptionInfoVoWithError:(NSError **)error;


/**
 取得訂閱帳戶限制
 
 @param error 回傳錯誤
 */
- (WCTRCGetAccountLimitResult *)subscribedAccountLimitWithError:(NSError **)error;


/**
 取得訂閱帳戶URL
 
 @param error 回傳錯誤
 */
- (WCTRCGetSubscriptionURLResponseResult *)subscriptionURLWithError:(NSError **)error;

/**
 試用天數升級
 
 @param error 回傳錯誤
 */
- (WCTRCBooleanResponseResult *)upgradeFromRetailWithError:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance CRM (Salesforce) Method


/// 取得最近瀏覽帳戶列表
/// @param error 回傳錯誤
- (WCTSalesForceSearchRecordResponseResult *)salesforceRecentViewAccountsWithError:(NSError **)error;



/// 取得關鍵字搜尋列表
/// @param keyword 關鍵字
/// @param error 回傳錯誤
- (WCTSalesForceSearchRecordResponseResult *)salesforceAccountsWithKeyword:(NSString *)keyword
                              error:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance QContact Method


/// 登入QContact
/// @param tokenVo 需求物件
/// @param error 回傳錯誤
- (WCTRCQContactzAuthInfoResponseResult *)qContactzLoginWithTokenVo:(WCTRCQContactzAuthTokenVo *)tokenVo
                        error:(NSError **)error;


/// 設定QContact 匯出方法
/// @param exportMode 匯出方法
/// @param error 回傳錯誤
- (WCTRCMessageResponseResult *)setQContactzExportMode:(NSString *)exportMode
                                                error:(NSError **)error;



/// QCONTACTZ 郵件寄送取得兩步驟驗證碼
/// @param basicInfo 需求物件
/// @param error 回傳錯誤
- (WCTRCQContactzEmailResponseResult *)setQContactzEmailSecurityCodeWithBasicInfo:(WCTRCAuthInfo *)basicInfo
                                          error:(NSError **)error;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance MyCard Method

/**
 * 取得我的電子名片
 * @param ID 指定我的名片ID
 * @param error 回傳錯誤
 * @return 回傳我的電子名片內容
 */
- (WCTRCMyCardInfoResponseResult *)myCardInfoWithID:(NSString *)ID error:(NSError **)error;


/**
 * 取得我的電子名片的名片圖
 * @param ID 指定我的名片ID
 * @param imageType 圖片類型 WCTRC_ImageType_Front, WCTRC_ImageType_Rear, WCTRC_ImageType_Logo
 * @param error 回傳錯誤
 * @return 回傳我的電子名片內容
 */
- (CPImage *)myCardImageWithID:(NSString *)ID
                     imageType:(NSString *)imageType
                         error:(NSError **)error;



/**
 * 更新我的電子名片
 * @param ID 指定我的名片ID
 * @param error 回傳錯誤
 * @return 回傳結果
 */

- (WCTRCMessageResponseResult *)upsertMyCardWithID:(NSString *)ID
                                          cardInfo:(WCTRCUpdateMyCardInfoRequest *)cardInfoRequest
                                             error:(NSError **)error;

/**
 * 取得我的電子名片設定
 * @param error 回傳錯誤
 * @return 回傳我的電子名片設定
 */

- (WCTRCMyCardSettingResponseResult *)myCardSettingWithError:(NSError **)error;

/**
 * 啟用我的名片分享
 * @param enable 啟用或關閉
 * @param ID 指定我的名片ID
 * @param error 回傳錯誤
 * @return 回傳我的電子名片設定
 */
- (WCTRCBooleanResponseResult *)myCardEnable:(BOOL)enable withID:(NSString *)ID error:(NSError **)error;

/**
 * 取得我的電子名片VCF
 * @param ID 指定我的名片ID
 * @param error 回傳錯誤
 * @return 回傳我的電子名片設定
 */
- (NSData *)myCardVCFWithID:(NSString *)ID error:(NSError **)error;


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance

/**
 * 是否支援帳號管理
 * @return 回傳支援帳號管理
 */
- (BOOL)isAccountManagerSupported;





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PCS Methods (精準校正)
/**
 * 取得PCS設定
 * @param error 回傳錯誤
 * @return WCTRCPCSSettingResponseResult
 */
- (WCTRCPCSSettingResponseResult *)pcsSettingWithError:(NSError **)error;

/**
 * 轉換PCS額度
 * @param request quota資訊
 * @param error 回傳錯誤
 * @return WCTRCPCSSettingResponseResult
 */
- (WCTRCMessageResponseResult *)transferPCSQuotaWithRequest:(WCTRCTransferQuotaRequest *)request error:(NSError **)error;




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - TSMC methods


/**
 台積電版本，取得匯出Salesforce Contact中沒有帳號的聯絡人
 @param accountInfo 註冊資訊
 @param error 回傳錯誤
 @return WCTRCMessageResponseResult  回傳執行結果
 */
- (WCTRCPrivateContactListResponseResult *)noAccountSFContactWithPageRequest:(WCTRCContactPageRequest *)pageRequest error:(NSError **)error;

/**
 台積電版本，註冊帳號
 @param accountInfo 註冊資訊
 @param error 回傳錯誤
 @return WCTRCMessageResponseResult  回傳執行結果
 */
- (WCTRCMessageResponseResult *)applyAccountWithAccountInfo:(WCTRCApplyAccountInfo *)accountInfo error:(NSError **)error;



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class path methods

+ (NSURL *)accountManagerURL;
@end
