//
//  PPRestClient.h
//  
//
//  Created by Howard on 2016/3/9.
//
//

#import <Foundation/Foundation.h>
#import "RestKit.h"
#import "PPRestObject.h"

// Define
#import "CrossPlatformDefine.h"
#import "PPRestClient+ParameterDefine.h"

@interface PPRestClient : NSObject

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property

@property (nonatomic, retain) NSDate *lastOperationDate;
@property (nonatomic, assign) dispatch_queue_t successCallbackQueue;
@property (nonatomic, assign) dispatch_queue_t failureCallbackQueue;
@property (nonatomic, assign) NSTimeInterval timeOutInterval;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

/**
 *  建構子
 *
 *  @param url 連結 server 網址
 *
 *  @return PPRestClient object
 */
- (id)initWithBaseURL:(NSURL *)url;

/**
 * 取得要使用的objectManager
 * @param type PPRCObjectManagerType_JSON，回傳requestSerializationMIMEType是json的objectManager；(body, query)
 *            PPRCObjectManagerType_URLEncoding, 回傳requestSerializationMIMEType是urlEncoded的objectManager; (formData)
 *             PPRCObjectManagerType_JSON_Image，回傳requestSerializationMIMEType是imageData的objectManager；(body, data)
 * @return RKObjectManager objectManager
 */
- (RKObjectManager *)objectManagerForType:(PPRCObjectManagerType)type;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance register Methods

/**
 *  註冊 Request Json 對應 Object Class 的關係
 *
 *  @param objectClass Request Object
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param rootKeyPath Request Object
 *  @param method Request Object
 *
 *  @return YES, 成功
 */
- (BOOL)addRequestDescriptorForObjectClass:(Class)objectClass
                           toObjectManager:(RKObjectManager *)objectManager
                               rootKeyPath:(NSString *)rootKeyPath
                                    method:(RKRequestMethod)method;

/**
 *  註冊 Request Json 對應 Object Class 的關係
 *
 *  @param objectClass Request Object
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *
 *  @return YES, 成功
 */
- (BOOL)addRequestDescriptorForObjectClass:(Class)objectClass
                           toObjectManager:(RKObjectManager *)objectManager;




/**
 *  註冊 Response Json 對應 Object Class 的關係
 *
 *  @param objectClass Request Object
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param method 這個對應的使用對象
 *  @param pathPattern 這個對應的使用對象
 *  @param keyPath 這個對應的使用對象
 *  @param statusCodes 這個對應的使用對象
 *
 *  @return YES, 成功
 */
- (BOOL)addResponseDescriptorForObjectClass:(Class)objectClass
                            toObjectManager:(RKObjectManager *)objectManager
                                     method:(RKRequestMethod)method
                                pathPattern:(NSString *)pathPattern
                                    keyPath:(NSString *)keyPath
                                statusCodes:(NSIndexSet *)statusCodes;


/**
 *  註冊 Response Json 對應 Object Class 的關係
 *
 *  @param objectClass Request Object
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param pathPattern 這個對應的使用對象
 *
 *  @return YES, 成功
 */
- (BOOL)addResponseDescriptorForObjectClass:(Class)objectClass
                            toObjectManager:(RKObjectManager *)objectManager
                                pathPattern:(NSString *)pathPattern;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance UnRegister Method


/**
 移除註冊對應

 *  @param objectClass Request Object
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param method 這個對應的使用對象
 *  @param pathPattern 這個對應的使用對象
 *  @param keyPath 這個對應的使用對象
 *  @param statusCodes 這個對應的使用對象
 *  @return YES, 成功
 */
- (BOOL)removeResponseDescriptorForObjectClass:(Class)objectClass
                               toObjectManager:(RKObjectManager *)objectManager
                                        method:(RKRequestMethod)method
                                   pathPattern:(NSString *)pathPattern
                                       keyPath:(NSString *)keyPath
                                   statusCodes:(NSIndexSet *)statusCodes;






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance GET Method

/**
 *  GET Method
 *
 *  @param object     自訂的 PPRestObject
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param path       API Name
 *  @param parameters HTTP Body Parameter
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 *
 */
- (RKObjectRequestOperation *)getObject:(id)object
                        byObjectManager:(RKObjectManager *)objectManager
                                   path:(NSString *)path
                             parameters:(NSDictionary *)parameters
                                success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;

/**
 *  GET Method
 *
 *  @param object     自訂的 PPRestObject
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param path       API Name
 *  @param parameters HTTP Body Parameter
 *  @param httpHeaderParameters HTTP Header 參數
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 *
 */
- (RKObjectRequestOperation *)getObject:(id)object
                        byObjectManager:(RKObjectManager *)objectManager
                                   path:(NSString *)path
                             parameters:(NSDictionary *)parameters
                   httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance POST Method

/**
 *  POST Method
 *
 *  @param object     自訂的 PPRestObject (如果參數很多，可用parameters, 也可以建成 PPRestObject 來轉換)
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param path       API Name
 *  @param parameters HTTP Body Parameter (如果Parameter type是form Data都用這個傳)
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 *
 */
- (RKObjectRequestOperation *)postObject:(id)object
                         byObjectManager:(RKObjectManager *)objectManager
                                    path:(NSString *)path
                              parameters:(NSDictionary *)parameters
                                 success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                 failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;


/**
 *  POST Method
 *
 *  @param object     自訂的 PPRestObject (如果參數很多，可用parameters, 也可以建成 PPRestObject 來轉換)
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param path       API Name
 *  @param parameters HTTP Body Parameter (如果Parameter type是form Data都用這個傳)
 *  @param httpHeaderParameters HTTP Header 參數
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 *
 */
- (RKObjectRequestOperation *)postObject:(id)object
                         byObjectManager:(RKObjectManager *)objectManager
                                    path:(NSString *)path
                              parameters:(NSDictionary *)parameters
                    httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                 success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                 failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;


/**
 *  POST Method （若 Body 塞的參數是 JSON，但不存在 Key 值)
 *
 *  @param path       API Name
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param parameterArray HTTP Body Parameter (NSArray)
 *  @param httpHeaderParameters HTTP Header 參數
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 */
- (RKObjectRequestOperation *)postWithPath:(NSString *)path
                           byObjectManager:(RKObjectManager *)objectManager
                            parameterArray:(NSArray *)parameterArray
                      httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                   success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                   failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;



- (RKObjectRequestOperation *)postWithPath:(NSString *)path
                           byObjectManager:(RKObjectManager *)objectManager
                                    object:(id)object
                      httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                   success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                   failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance DELETE Method

/**
 *  DELETE Method
 *
 *  @param object     自訂的 PPRestObject (如果參數很多，可用parameters, 也可以建成 PPRestObject 來轉換)
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param path       API Name
 *  @param parameters HTTP Body Parameter (如果Parameter type是form Data都用這個傳)
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 *
 */
- (RKObjectRequestOperation *)deleteObject:(id)object
                           byObjectManager:(RKObjectManager *)objectManager
                                      path:(NSString *)path
                                parameters:(NSDictionary *)parameters
                                   success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                   failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;


/**
 *  DELETE Method
 *
 *  @param object     自訂的 PPRestObject (如果參數很多，可用parameters, 也可以建成 PPRestObject 來轉換)
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param path       API Name
 *  @param parameters HTTP Body Parameter (如果Parameter type是form Data都用這個傳)
 *  @param httpHeaderParameters HTTP Header 參數
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 *
 */
- (RKObjectRequestOperation *)deleteObject:(id)object
                           byObjectManager:(RKObjectManager *)objectManager
                                      path:(NSString *)path
                                parameters:(NSDictionary *)parameters
                      httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                   success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                   failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance PUT Method


/**
*  PUT Method
*
*  @param object     自訂的 PPRestObject (如果參數很多，可用parameters, 也可以建成 PPRestObject 來轉換)
*  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
*  @param path       API Name
*  @param parameters HTTP Body Parameter (如果Parameter type是form Data都用這個傳)
*  @param success    成功的回傳 function
*  @param failure    失敗的回傳 function
*
*  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
*
*/
- (RKObjectRequestOperation *)putObject:(id)object
                        byObjectManager:(RKObjectManager *)objectManager
                                   path:(NSString *)path
                             parameters:(NSDictionary *)parameters
                                success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;




/**
 *  PUT Method
 *
 *  @param object     自訂的 PPRestObject (如果參數很多，可用parameters, 也可以建成 PPRestObject 來轉換)
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param path       API Name
 *  @param parameters HTTP Body Parameter (如果Parameter type是form Data都用這個傳)
 *  @param httpHeaderParameters HTTP Header 參數
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 *
 */
- (RKObjectRequestOperation *)putObject:(id)object
                        byObjectManager:(RKObjectManager *)objectManager
                                   path:(NSString *)path
                             parameters:(NSDictionary *)parameters
                   httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;


- (RKObjectRequestOperation *)putWithPath:(NSString *)path
                          byObjectManager:(RKObjectManager *)objectManager
                                 httpBody:(NSData *)httpBody
                     httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                  success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                  failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;


- (RKObjectRequestOperation *)putWithPath:(NSString *)path
                          byObjectManager:(RKObjectManager *)objectManager
                                   object:(id)object
                     httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                  success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                  failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance PATCH Method


/**
 *  PATCH Method
 *
 *  @param object     自訂的 PPRestObject (如果參數很多，可用parameters, 也可以建成 PPRestObject 來轉換)
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param path       API Name
 *  @param parameters HTTP Body Parameter (如果Parameter type是form Data都用這個傳)
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 *
 */
- (RKObjectRequestOperation *)patchObject:(id)object
                          byObjectManager:(RKObjectManager *)objectManager
                                     path:(NSString *)path
                               parameters:(NSDictionary *)parameters
                                  success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                  failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;




/**
 *  PATCH Method
 *
 *  @param object     自訂的 PPRestObject (如果參數很多，可用parameters, 也可以建成 PPRestObject 來轉換)
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param path       API Name
 *  @param parameters HTTP Body Parameter (如果Parameter type是form Data都用這個傳)
 *  @param httpHeaderParameters HTTP Header 參數
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 *
 */
- (RKObjectRequestOperation *)patchObject:(id)object
                          byObjectManager:(RKObjectManager *)objectManager
                                     path:(NSString *)path
                               parameters:(NSDictionary *)parameters
                     httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                  success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                  failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;

/**
 *  PATCH Method
 *
 *  @param object     自訂的 PPRestObject (如果參數很多，可用parameters, 也可以建成 PPRestObject 來轉換)
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param path       API Name
 *  @param parameters HTTP Body Parameter (如果Parameter type是form Data都用這個傳)
 *  @param httpHeaderParameters HTTP Header 參數
 *  @param removeNullsInBody 是否移除http body中Null的參數
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 *
 */
- (RKObjectRequestOperation *)patchObject:(id)object
                          byObjectManager:(RKObjectManager *)objectManager
                                     path:(NSString *)path
                               parameters:(NSDictionary *)parameters
                     httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                        removeNullsInBody:(BOOL)removeNullsInBody
                                  success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                  failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;


- (RKObjectRequestOperation *)patchWithPath:(NSString *)path
                            byObjectManager:(RKObjectManager *)objectManager
                                   httpBody:(NSData *)httpBody
                       httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                    success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                    failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;


- (RKObjectRequestOperation *)patchWithPath:(NSString *)path
                            byObjectManager:(RKObjectManager *)objectManager
                                     object:(id)object
                       httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                    success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                    failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Batch Method

/**
 *  透過 multiPartForm 方式傳送封包 (通常用在附檔)
 *
 *  @param object               自訂的 PPRestObject
 *  @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 *  @param method               GET or POST
 *  @param path                 API Name
 *  @param data                 資料
 *  @param name                 名稱 (看伺服器定義)
 *  @param fileName             檔案名稱 (儲存檔案名稱)
 *  @param mimeType             檔案型態
 *  @param parameters           HTTP Body Parameter
 *  @param httpHeaderParameters HTTP Header Parameter
 *  @param success    成功的回傳 function
 *  @param failure    失敗的回傳 function
 *
 *  @return RKObjectRequestOperation isn't nil 代表成功送出 Request
 */
- (RKObjectRequestOperation *)multipartFormRequestWithObject:(id)object
                                             byObjectManager:(RKObjectManager *)objectManager
                                                      method:(RKRequestMethod)method
                                                        path:(NSString *)path
                                                        data:(NSData *)data
                                                        name:(NSString *)name
                                                    fileName:(NSString *)fileName
                                                    mimeType:(NSString *)mimeType
                                                  parameters:(NSDictionary *)parameters
                                        httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                                     success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                                     failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure;






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instace Image or Data Method

/**
 透過 Get 方式，取得圖像

 @param object 自訂的 PPRestObject
 @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 @param path API Name
 @param parameters HTTP Body Parameters
 @param httpHeaderParameters HTTP Head Parameters
 @param success 成功回傳
 @param failure 失敗回傳
 @return AFRKImageRequestOperation isn't nil 代表成功送出 Request
 */
- (AFRKImageRequestOperation *)getImageWithObject:(id)object
                                    objectManager:(RKObjectManager *)objectManager
                                             path:(NSString *)path
                                       parameters:(NSDictionary *)parameters
                             httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                          success:(void (^)(NSURLRequest *request, NSHTTPURLResponse *response, CPImage *image))success
                                          failure:(void (^)(NSURLRequest *request, NSHTTPURLResponse *response, NSError *error))failure;

/**
 透過 Get 方式，取得資料(不轉為json)
 
 @param object 自訂的 PPRestObject
 @param objectManager 用來傳送request的實體，從|objectManagerForType:|取得
 @param path API Name
 @param parameters HTTP Body Parameters
 @param httpHeaderParameters HTTP Head Parameters
 @param success 成功回傳
 @param failure 失敗回傳
 @return AFRKHTTPRequestOperation isn't nil 代表成功送出 Request
 */
- (AFRKHTTPRequestOperation *)getDataWithObject:(id)object
                                  objectManager:(RKObjectManager *)objectManager
                                           path:(NSString *)path
                                     parameters:(NSDictionary *)parameters
                           httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                        success:(void (^)(AFRKHTTPRequestOperation *operation, NSData *responseData))success
                                        failure:(void (^)(AFRKHTTPRequestOperation *operation, NSError *error))failure;




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance IP Method

- (NSString *)ipAddress;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Operation Method

- (void)cancellAllRestOperation;

- (void)waitAllRestOperaitonAreFinished;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Image Data Method


/**
 取得圖像資料

 @param image 圖像
 @param mineType 類型 PPRestClientImageType_Png or PPRestClientImageType_Jpg
 @return 圖像資料
 */
- (NSData *)dataForImage:(CPImage *)image mineType:(NSString *)mineType;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class AFNetworking Error Releated Method

/**
 * 從restkit回傳的Error中取得response
 * @param error RestKit的Error
 * @return NSHTTPURLResponse, 如果不存在回傳nil
 */
+ (NSHTTPURLResponse *)responseFromAFRKNetworkingError:(NSError *)error;


/**
 * 從restkit回傳的Error中取得status code
 * @param error RestKit的Error
 * @return NSInteger status code from server, 不存在則回傳NSNotFound
 */
+ (NSInteger)statusCodeFromAFRKNetworkingError:(NSError *)error;


/**
 * 從restkit回傳的Error中取得domain
 * @param error RestKit的Error
 * @return NSString Domain name
 */
+ (NSString *)domainFromAFRKNetworkingError:(NSError *)error;


@end
