//
//  UIImage+Additions.m
//  
//
//  Created by Mike on 13/5/2.
//  
//

#import "UIImage+Additions.h"
#import "UIImage+BITMAPPTR.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation UIImage (Additions)


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private Methods

//===============================================================================
//
//===============================================================================
- (CGContextRef)newBitmapContextWithSize:(CGSize)newSize
{
    CGContextRef    context = nil;
    CGBitmapInfo    bitmapInfo = CGImageGetBitmapInfo(self.CGImage);
    CGColorSpaceRef colorSpace = CGImageGetColorSpace(self.CGImage);
    
    NSUInteger bitsPerComponent = 8;
    NSUInteger componentCount = 0;
    NSUInteger bytesPerRow = 0;
    
    
    //////////////////////////////////////////////////

    if(colorSpace!=NULL)
    {
        if(((bitmapInfo & kCGImageAlphaPremultipliedLast) == kCGImageAlphaPremultipliedLast)    ||
           ((bitmapInfo & kCGImageAlphaPremultipliedFirst) == kCGImageAlphaPremultipliedFirst)  ||
           ((bitmapInfo & kCGImageAlphaLast) == kCGImageAlphaLast)                              ||
           ((bitmapInfo & kCGImageAlphaFirst) == kCGImageAlphaFirst)                            ||
           ((bitmapInfo & kCGImageAlphaNoneSkipLast) == kCGImageAlphaNoneSkipLast)              ||
           ((bitmapInfo & kCGImageAlphaNoneSkipFirst) == kCGImageAlphaNoneSkipFirst)            )
        {
            componentCount = 4;
//            if (((bitmapInfo & kCGImageAlphaLast) == kCGImageAlphaLast))
//            {
//                bitmapInfo = kCGBitmapAlphaInfoMask&kCGImageAlphaPremultipliedLast;
//            }
            // ios 9 kCGBitmapByteOrder32Little可以產生
            // ios 10之後會回傳kCGBitmapByteOrder16Little, 有可能會造成無法產生bitmapcontext所以只取kCGBitmapAlphaInfoMask
            bitmapInfo = kCGBitmapAlphaInfoMask&kCGImageAlphaPremultipliedLast;
        }
        else if((bitmapInfo & kCGImageAlphaOnly) == kCGImageAlphaOnly)
        {
            componentCount = 1;
        }
        else
        {
            componentCount = 3;
        }
        
        //////////////////////////////////////////////////
        
        bytesPerRow = [UIImage bytesPerRowWithWidth:newSize.width bitCount:bitsPerComponent*componentCount];
        context = CGBitmapContextCreate(NULL, newSize.width, newSize.height, bitsPerComponent, bytesPerRow, colorSpace, bitmapInfo);
    }
    
    return context;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods

//================================================================================
// 配合macOS加入的函式，取得影像真正大小，在iOS和size是相同的。
//================================================================================
- (CGSize)actualSize
{
    return self.size;
}


//===============================================================================
//
//===============================================================================
- (UIImage *)imageAdjustWithBrightness:(NSInteger)brightness contrast:(NSInteger)contrast
{
    UIImage *image = nil;
    
    //////////////////////////////////////////////////
    
    BITMAPPTR inputBitmapPTR;
    memset(&inputBitmapPTR, 0, sizeof(BITMAPPTR));
    
    if([UIImage createBitmapPTR:&inputBitmapPTR image:(self.imageOrientation==UIImageOrientationUp?self:[self imageRotatedByDegrees:0])]==YES)
    {
        Byte lookupTable[256];
        
        if(contrast>0)
        {
            for(int index=0; index<256; index++)
            {
                lookupTable[index] = MAX(0, MIN(255, (25500*index+25500*brightness-32385*contrast)/(25500-254*contrast)));
            }
        }
        else
        {
            for(int index=0; index<256; index++)
            {
                lookupTable[index] = MAX(0, MIN(255, (256*(100+contrast)*(index+brightness)-32640*contrast)/25500));
            }
        }
        
        //////////////////////////////////////////////////
        
        NSUInteger bytesPerRow = [UIImage bytesPerRowWithWidth:inputBitmapPTR.pHeader->biWidth bitCount:inputBitmapPTR.pHeader->biBitCount];
        
        switch(inputBitmapPTR.pHeader->biBitCount)
        {
            case 8:
            {
                for(long row=0; row<labs(inputBitmapPTR.pHeader->biHeight); row++)
                {
                    for(long column=0; column<inputBitmapPTR.pHeader->biWidth; column++)
                    {
                        NSUInteger index = (row*bytesPerRow)+column;
                        
                        inputBitmapPTR.pBmp[index] = lookupTable[inputBitmapPTR.pBmp[index]];
                    }
                }
                
                break;
            }
            case 24:
            case 32:
            {
                for(long row=0; row<labs(inputBitmapPTR.pHeader->biHeight); row++)
                {
                    for(long column=0; column<inputBitmapPTR.pHeader->biWidth; column++)
                    {
                        NSUInteger index = (row*bytesPerRow)+(column*3);
                        
                        inputBitmapPTR.pBmp[index]      = lookupTable[inputBitmapPTR.pBmp[index]];
                        inputBitmapPTR.pBmp[index+1]    = lookupTable[inputBitmapPTR.pBmp[index+1]];
                        inputBitmapPTR.pBmp[index+2]    = lookupTable[inputBitmapPTR.pBmp[index+2]];
                    }
                }
                
                break;
            }
            default:
            {
                break;
            }
        }
        
        //////////////////////////////////////////////////
        
        image = [UIImage imageWithBitmapPTR:&inputBitmapPTR];
        
        //////////////////////////////////////////////////
        
        [UIImage releaseBitmapPTR:&inputBitmapPTR];
    }
    
    //////////////////////////////////////////////////
    
    return image;
}

//===============================================================================
//
//===============================================================================
- (UIImage *)imageCroppedByRect:(CGRect)rect
{
    UIImage *image  = nil;
    
    //////////////////////////////////////////////////
    
    CGImageRef cgImage = CGImageCreateWithImageInRect([self imageRotatedByDegrees:0 scalingMaxLength:0].CGImage, rect);
    if(cgImage!=NULL)
    {
     	image = [UIImage imageWithCGImage:cgImage];
        
        CGImageRelease(cgImage);
    }
    
    //////////////////////////////////////////////////
    
    return 	image;
}

//===============================================================================
//
//===============================================================================
- (UIImage *)imageReflectionByFraction:(CGFloat)fraction
{
	UIImage *image = nil;
	
	//////////////////////////////////////////////////
    
	//倒影高度
	NSInteger reflectionHeight = self.size.height*fraction;
	
	//////////////////////////////////////////////////
    
	//產生灰階的色彩空間
    CGColorSpaceRef colorSpaceOfGray = CGColorSpaceCreateDeviceGray();
	if(colorSpaceOfGray!=NULL)
	{
		//建立倒影遮罩容器
		CGContextRef contextOfReflectionMask = CGBitmapContextCreate(NULL, 1, reflectionHeight, 8, 1, colorSpaceOfGray, kCGBitmapAlphaInfoMask&kCGImageAlphaNone);
		if(contextOfReflectionMask!=NULL)
		{
			//建立漸層物件
			CGFloat colors[] = {0.0, 1.0, 1.0, 1.0};
			CGGradientRef reflectionMaskGradient = CGGradientCreateWithColorComponents(colorSpaceOfGray, colors, NULL, 2);
			if(reflectionMaskGradient!=NULL)
			{
				CGPoint gradientPointStart	= CGPointMake(0, reflectionHeight);
				CGPoint gradientPointEnd	= CGPointZero;
				
				//建立倒影線性漸層
				CGContextDrawLinearGradient(contextOfReflectionMask, reflectionMaskGradient, gradientPointStart, gradientPointEnd, kCGGradientDrawsAfterEndLocation);
				
				CGGradientRelease(reflectionMaskGradient);
			}
			
			//////////////////////////////////////////////////
			
			//疊上透明度50%的黑色矩形
			CGContextSetGrayFillColor(contextOfReflectionMask, 0.0, 0.5);
			CGContextFillRect(contextOfReflectionMask, CGRectMake(0, 0, 1, reflectionHeight));
			
			//////////////////////////////////////////////////
			
			CGImageRef cgImageOfReflectionMask = CGBitmapContextCreateImage(contextOfReflectionMask);
			if(cgImageOfReflectionMask!=NULL)
			{
				CGImageRef cgImageOfReflection = CGImageCreateWithMask(self.CGImage, cgImageOfReflectionMask);
				if(cgImageOfReflection!=NULL)
				{
                    CGColorSpaceRef colorSpaceOfRGB = CGColorSpaceCreateDeviceRGB();
                    if(colorSpaceOfRGB!=NULL)
                    {
                        CGContextRef context = CGBitmapContextCreate(NULL,
                                                                     self.size.width,
                                                                     self.size.height+reflectionHeight,
                                                                     8,
                                                                     [UIImage bytesPerRowWithWidth:self.size.width bitCount:32],
                                                                     colorSpaceOfRGB,
                                                                     kCGBitmapAlphaInfoMask&kCGImageAlphaPremultipliedLast);
                        if(context!=NULL)
                        {
                            //將座標系改為第四象限
                            CGContextScaleCTM(context, 1, -1);
                            
                            //先畫出原圖
                            CGContextDrawImage(context, CGRectMake(0, 0, self.size.width, -self.size.height), self.CGImage);
                            
                            //再畫倒影圖
                            CGContextDrawImage(context, CGRectMake(0, self.size.height, self.size.width, reflectionHeight), cgImageOfReflection);
                         
                            //////////////////////////////////////////////////
                            
                            CGImageRef cgImage = CGBitmapContextCreateImage(context);
                            if(cgImage!=NULL)
                            {
                                image = [UIImage imageWithCGImage:cgImage];
                                
                                CGImageRelease(cgImage);
                            }
                            
                            //////////////////////////////////////////////////
            
                            CGContextRelease(context);
                        }
                        
                        CGColorSpaceRelease(colorSpaceOfRGB);
                    }
					
					CGImageRelease(cgImageOfReflection);
				}
				
				CGImageRelease(cgImageOfReflectionMask);
			}
			
			CGContextRelease(contextOfReflectionMask);
		}
		
		CGColorSpaceRelease(colorSpaceOfGray);
	}
	
	return image;
}

//===============================================================================
//
//===============================================================================
- (UIImage *)imageRotatedByDegrees:(CGFloat)degrees scalingMaxLength:(CGFloat)maxLength
{
    UIImage *image = nil;
    
    do
    {
        // !! 比maxLength小的圖不用縮放
        if (self.size.width<=maxLength&&
            self.size.height<=maxLength)
        {
            maxLength = 0;
        }
        
        CGSize scalingSize = [self sizeRotatedByDegrees:degrees scalingToMaxLength:maxLength];
        
        //////////////////////////////////////////////////
        
        if(CGSizeEqualToSize(self.size, scalingSize) && self.imageOrientation==UIImageOrientationUp && ((int)degrees)%360==0)
        {
            image = [[self retain] autorelease];
            break;
        }
        
        //////////////////////////////////////////////////
        
        CGContextRef context = [self newBitmapContextWithSize:scalingSize];
        
        if(context!=NULL)
        {
            //將旋轉點移到中心點
            CGContextTranslateCTM(context, scalingSize.width/2, scalingSize.height/2);
            
            //////////////////////////////////////////////////
            
            switch(self.imageOrientation)
            {
                case UIImageOrientationLeft:
                {
                    CGContextRotateCTM(context, radiansFromDegrees(90-degrees));
                    
                    if(fabs(degrees)==90.0f || fabs(degrees)==270.0f)
                    {
                        CGContextDrawImage(context, CGRectMake(-scalingSize.width/2, -scalingSize.height/2, scalingSize.width, scalingSize.height), self.CGImage);
                        
                    }
                    else
                    {
                        CGContextDrawImage(context, CGRectMake(-scalingSize.height/2, -scalingSize.width/2, scalingSize.height, scalingSize.width), self.CGImage);
                    }
                    
                    break;
                }
                case UIImageOrientationDown:
                {
                    CGContextRotateCTM(context, radiansFromDegrees(180-degrees));
                    
                    if(fabs(degrees)==90.0f || fabs(degrees)==270.0f)
                    {
                        CGContextDrawImage(context, CGRectMake(-scalingSize.height/2, -scalingSize.width/2, scalingSize.height, scalingSize.width), self.CGImage);
                    }
                    else
                    {
                        CGContextDrawImage(context, CGRectMake(-scalingSize.width/2, -scalingSize.height/2, scalingSize.width, scalingSize.height), self.CGImage);
                    }
                    
                    break;
                }
                case UIImageOrientationRight:
                {
                    CGContextRotateCTM(context, radiansFromDegrees(270-degrees));
                    
                    if(fabs(degrees)==90.0f || fabs(degrees)==270.0f)
                    {
                        CGContextDrawImage(context, CGRectMake(-scalingSize.width/2, -scalingSize.height/2, scalingSize.width, scalingSize.height), self.CGImage);
                        
                    }
                    else
                    {
                        CGContextDrawImage(context, CGRectMake(-scalingSize.height/2, -scalingSize.width/2, scalingSize.height, scalingSize.width), self.CGImage);
                    }
                    
                    break;
                }
                case UIImageOrientationUp:
                default:
                {
                    CGContextRotateCTM(context, radiansFromDegrees(-degrees));
                    
                    if(fabs(degrees)==90.0f || fabs(degrees)==270.0f)
                    {
                        CGContextDrawImage(context, CGRectMake(-scalingSize.height/2, -scalingSize.width/2, scalingSize.height, scalingSize.width), self.CGImage);
                    }
                    else
                    {
                        CGContextDrawImage(context, CGRectMake(-scalingSize.width/2, -scalingSize.height/2, scalingSize.width, scalingSize.height), self.CGImage);
                    }
                    
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            
            CGImageRef cgImage = CGBitmapContextCreateImage(context);
            
            if(cgImage!=NULL)
            {
                image = [UIImage imageWithCGImage:cgImage];
                
                CGImageRelease(cgImage);
            }
            
            //////////////////////////////////////////////////
            
            CGContextRelease(context);
        }
        
    }while(0);
    
    return image;
}

//===============================================================================
//
//===============================================================================
- (UIImage *)imageRotatedByDegrees:(CGFloat)degrees
{
    return [self imageRotatedByDegrees:degrees scalingMaxLength:0];
}

//===============================================================================
//
//===============================================================================
- (UIImage *)imageRotatedByRadians:(CGFloat)radians
{
    return [self imageRotatedByDegrees:degreesFromRadians(radians)];
}

//===============================================================================
//
//===============================================================================
- (UIImage *)imageScalingMaxLength:(CGFloat)maxLength
{
    return [self imageRotatedByDegrees:0 scalingMaxLength:maxLength];
}

//===============================================================================
//
//===============================================================================
- (CGSize)sizeRotatedByDegrees:(CGFloat)degrees scalingToFitSize:(CGSize)fitSize
{
    CGSize size;
    
    //////////////////////////////////////////////////
    
    if(CGSizeEqualToSize(fitSize, CGSizeZero)==YES)
    {
        fitSize.width   = MAX(self.size.width, self.size.height);
        fitSize.height  = fitSize.width;
    }
    
    //////////////////////////////////////////////////
    
    if(fabs(degrees)==90.0f || fabs(degrees)==270.0f)
    {
        if((self.size.height/self.size.width)>(fitSize.width/fitSize.height))
        {
            size = CGSizeMake(fitSize.width, fitSize.width*self.size.width/self.size.height);
        }
        else
        {
            size = CGSizeMake(fitSize.height*self.size.height/self.size.width, fitSize.height);
        }
    }
    else
    {
        if((self.size.width/self.size.height)>(fitSize.width/fitSize.height))//寬比較大
        {
            size = CGSizeMake(fitSize.width, fitSize.width*self.size.height/self.size.width);
        }
        else
        {
            size = CGSizeMake(fitSize.height*self.size.width/self.size.height, fitSize.height);
        }
    }
    
    //////////////////////////////////////////////////
    
	size.width	= floor(size.width);
	size.height	= floor(size.height);
    
    //////////////////////////////////////////////////
    
    return size;
}

//===============================================================================
//
//===============================================================================
- (CGSize)sizeRotatedByDegrees:(CGFloat)degrees scalingToMaxLength:(CGFloat)maxLength
{
    return [self sizeRotatedByDegrees:degrees scalingToFitSize:CGSizeMake(maxLength, maxLength)];
}

//===============================================================================
//
//===============================================================================
- (NSData *)dataOfRGBA
{
    NSData *data = nil;
    
    //////////////////////////////////////////////////
    
    CGColorSpaceRef colorSpace = CGColorSpaceCreateDeviceRGB();
    if(colorSpace!=NULL)
    {
        CGContextRef context = CGBitmapContextCreate(NULL,
                                                     self.size.width,
                                                     self.size.height,
                                                     8,
                                                     [UIImage bytesPerRowWithWidth:self.size.width bitCount:32],
                                                     colorSpace,
                                                     kCGBitmapAlphaInfoMask&kCGImageAlphaPremultipliedLast);
        if(context!=NULL)
        {
            CGContextScaleCTM(context, 1, -1);
            CGContextDrawImage(context, CGRectMake(0, 0, self.size.width, -self.size.height), self.CGImage);
            
            Byte *bytes = (Byte *)CGBitmapContextGetData(context);
            if(bytes!=NULL)
            {
                data = [NSData dataWithBytes:bytes
                                      length:CGBitmapContextGetHeight(context)*CGBitmapContextGetBytesPerRow(context)];
            }
            
            CGContextRelease(context);
        }
        
        CGColorSpaceRelease(colorSpace);
    }
    
    //////////////////////////////////////////////////
    
    return data;
}


//==============================================================================
//
//==============================================================================
- (NSData *)jpegRepresentationWithCompressQuality:(CGFloat)compressQuality
{
    return UIImageJPEGRepresentation(self, compressQuality);
}


//==============================================================================
//
//==============================================================================
- (NSData *)pngRepresentation
{
    return UIImagePNGRepresentation(self);
}


//===============================================================================
//
//===============================================================================
- (UIImage *)imageHorzExpandedWithLeftCap:(CGFloat)leftCap rightCap:(CGFloat)rightCap newWidth:(CGFloat)newWidth
{
    UIImage *newImage = nil;
    
    //////////////////////////////////////////////////
    // split original image
    
    CGFloat     srcHeight = self.size.height;
    CGImageRef  srcImageRef = [self CGImage];
    CGImageRef  splitImageRef[3];
    
    
    // left
    splitImageRef[0] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(0, 0, leftCap, srcHeight));
    
    // middle
    splitImageRef[1] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(leftCap, 0, self.size.width-leftCap-rightCap, srcHeight));
    
    // right
    splitImageRef[2] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(self.size.width-rightCap, 0, rightCap, srcHeight));
    
    
    //////////////////////////////////////////////////
    // create and draw new image

    CGContextRef context = [self newBitmapContextWithSize:CGSizeMake(newWidth, srcHeight)];

    if(context)
    {
        // left
        CGContextDrawImage(context, CGRectMake(0, 0, leftCap, srcHeight), splitImageRef[0]);
        
        // middle
        CGContextDrawImage(context, CGRectMake(leftCap, 0, newWidth-leftCap-rightCap, srcHeight), splitImageRef[1]);
        
        // right
        CGContextDrawImage(context, CGRectMake(newWidth-rightCap, 0, rightCap, srcHeight), splitImageRef[2]);
        
        CGImageRef imageRef = CGBitmapContextCreateImage(context);
        
        if(imageRef)
        {
            newImage = [UIImage imageWithCGImage:imageRef];
            CGImageRelease(imageRef);
        }
        
        CGContextRelease(context);
    }
    
    //////////////////////////////////////////////////
    
    for(int i=0; i<3; i++)
    {
        CGImageRelease(splitImageRef[i]);
    }
    
    
	return newImage;
}


//===============================================================================
//
//===============================================================================
- (UIImage *)imageExpandedWithFixedCornerCap:(CGFloat)fixedCornerCap newSize:(CGSize)newSize
{
    UIImage *newImage = nil;
    
    //////////////////////////////////////////////////
    // split original image

    CGSize      srcSize = self.size;
    CGImageRef  srcImageRef = [self CGImage];
    CGImageRef  splitImageRef[9];
    
    
    // top-left
    splitImageRef[0] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(0, 0, fixedCornerCap, fixedCornerCap));
    
    // top-middle
    splitImageRef[1] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(fixedCornerCap, 0, srcSize.width-2*fixedCornerCap, fixedCornerCap));
    
    // top-right
    splitImageRef[2] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(srcSize.width-fixedCornerCap, 0, fixedCornerCap, fixedCornerCap));
    
    // middle-left
    splitImageRef[3] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(0, fixedCornerCap, fixedCornerCap, srcSize.height-2*fixedCornerCap));
    
    // middle-middle
    splitImageRef[4] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(fixedCornerCap, fixedCornerCap, srcSize.width-2*fixedCornerCap, srcSize.height-2*fixedCornerCap));
    
    // middle-right
    splitImageRef[5] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(srcSize.width-fixedCornerCap, fixedCornerCap, fixedCornerCap, srcSize.height-2*fixedCornerCap));
    
    // bottom-left
    splitImageRef[6] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(0, srcSize.height-fixedCornerCap, fixedCornerCap, fixedCornerCap));
    
    // bottom-middle
    splitImageRef[7] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(fixedCornerCap, srcSize.height-fixedCornerCap, srcSize.width-2*fixedCornerCap, fixedCornerCap));
    
    // bottom-right
    splitImageRef[8] = CGImageCreateWithImageInRect(srcImageRef, CGRectMake(srcSize.width-fixedCornerCap, srcSize.height-fixedCornerCap, fixedCornerCap, fixedCornerCap));
    
    
    //////////////////////////////////////////////////
    // create and draw new image
    
    CGContextRef context = [self newBitmapContextWithSize:newSize];
    
    if(context)
    {
        // top-left
        CGContextDrawImage(context, CGRectMake(0, newSize.height-fixedCornerCap, fixedCornerCap, fixedCornerCap), splitImageRef[0]);
        
        // top-middle
        CGContextDrawImage(context, CGRectMake(fixedCornerCap, newSize.height-fixedCornerCap, newSize.width-2*fixedCornerCap, fixedCornerCap), splitImageRef[1]);
        
        // top-right
        CGContextDrawImage(context, CGRectMake(newSize.width-fixedCornerCap, newSize.height-fixedCornerCap, fixedCornerCap, fixedCornerCap), splitImageRef[2]);
        
        // middle-left
        CGContextDrawImage(context, CGRectMake(0, fixedCornerCap, fixedCornerCap, newSize.height-2*fixedCornerCap), splitImageRef[3]);
        
        // middle-middle
        CGContextDrawImage(context, CGRectMake(fixedCornerCap, fixedCornerCap, newSize.width-2*fixedCornerCap, newSize.height-2*fixedCornerCap), splitImageRef[4]);
        
        // middle-right
        CGContextDrawImage(context, CGRectMake(newSize.width-fixedCornerCap, fixedCornerCap, fixedCornerCap, newSize.height-2*fixedCornerCap), splitImageRef[5]);
        
        // bottom-left
        CGContextDrawImage(context, CGRectMake(0, 0, fixedCornerCap, fixedCornerCap), splitImageRef[6]);
        
        // bottom-middle
        CGContextDrawImage(context, CGRectMake(fixedCornerCap, 0, newSize.width-2*fixedCornerCap, fixedCornerCap), splitImageRef[7]);
        
        // bottom-right
        CGContextDrawImage(context, CGRectMake(newSize.width-fixedCornerCap, 0, fixedCornerCap, fixedCornerCap), splitImageRef[8]);
        
        CGImageRef imageRef = CGBitmapContextCreateImage(context);
        
        if(imageRef)
        {
            newImage = [UIImage imageWithCGImage:imageRef];
            CGImageRelease(imageRef);
        }
        
        CGContextRelease(context);
    }
    
    
    //////////////////////////////////////////////////

    for(int i=0; i<9; i++)
    {
        CGImageRelease(splitImageRef[i]);
    }
	
	return newImage;
}


//===============================================================================
// 建立局部影像
//===============================================================================
- (UIImage *)imageInRect:(CGRect)rect
{
    UIImage *newImage = nil;
	CGImageRef sourceImageRef = [self CGImage];
    
    if(sourceImageRef)
    {
        CGImageRef newImageRef = CGImageCreateWithImageInRect(sourceImageRef, rect);
        
        if(newImageRef)
        {
            newImage = [UIImage imageWithCGImage:newImageRef];
            CGImageRelease(newImageRef);
        }
    }
	
	return newImage;
}


//===============================================================================
//
//===============================================================================
- (UIImage *)imageByAdjustOrientationWithMaxLength:(NSInteger)maxLength
{
	UIImageOrientation	imageOrientation = self.imageOrientation;
	CGContextRef		bitmap = nil;
	CGImageRef			imageRef = [self CGImage];
	CGSize				targetSize = self.size;
	CGSize				drawSize;
    UIImage             *returnImage = self;
	
    
    //////////////////////////////////////////////////
	// calculate new size

	if(maxLength != 0)
	{
		if (self.size.width > self.size.height)
		{
			if(self.size.width <= maxLength)
			{
				targetSize = self.size;
			}
			else
			{
				targetSize.width = maxLength;
				targetSize.height = (NSInteger)(maxLength * self.size.height / self.size.width);
			}
		}
		else
		{
			if(self.size.height <= maxLength)
			{
				targetSize = self.size;
			}
			else
			{
				targetSize.height = maxLength;
				targetSize.width = (NSInteger)(maxLength * self.size.width / self.size.height);
			}
		}
	}
    
	
    //////////////////////////////////////////////////
    // generate new image

    bitmap = [self newBitmapContextWithSize:targetSize];
    
    if(bitmap)
    {
        switch(imageOrientation)
        {
            case UIImageOrientationLeft:
            {
                CGContextRotateCTM(bitmap, radiansFromDegrees(90));
                CGContextTranslateCTM (bitmap, 0, -targetSize.width);
                
                drawSize.width = targetSize.height;
                drawSize.height = targetSize.width;
                break;
            }
                
            case UIImageOrientationDown:
            {
                CGContextRotateCTM(bitmap, radiansFromDegrees(180));
                CGContextTranslateCTM(bitmap, -targetSize.width, -targetSize.height);
                
                drawSize = targetSize;
                break;
            }
                
            case UIImageOrientationRight:
            {
                CGContextRotateCTM(bitmap, radiansFromDegrees(270));
                CGContextTranslateCTM (bitmap, -targetSize.height, 0);
                
                drawSize.width = targetSize.height;
                drawSize.height = targetSize.width;
                break;
            }
                
            default :
            {
                if(CGSizeEqualToSize(self.size, targetSize))
                {
                    // 大小方向一樣就不用轉了
                    return self;
                }
                else
                {
                    drawSize = targetSize;
                }
            }
        }
        
        CGContextDrawImage(bitmap, CGRectMake(0, 0, drawSize.width, drawSize.height), imageRef);
        
        CGImageRef imageRef = CGBitmapContextCreateImage(bitmap);
        
        if(imageRef)
        {
            returnImage = [UIImage imageWithCGImage:imageRef];
            CGImageRelease(imageRef);
        }
        
        CGContextRelease(bitmap);
    }
    
	return returnImage;
}


//================================================================================
//
//================================================================================
- (UIImage *)removeWhiteBackgroundColor
{
  CGImageRef rawImageRef = self.CGImage;

  const CGFloat colorMasking[6] = {250, 255, 250, 255, 250, 255};

  UIGraphicsBeginImageContext(self.size);
  
    CGImageRef maskedImageRef=CGImageCreateWithMaskingColors(rawImageRef, colorMasking);
  {
    //if in iphone
    CGContextTranslateCTM(UIGraphicsGetCurrentContext(), 0.0, self.size.height);
    CGContextScaleCTM(UIGraphicsGetCurrentContext(), 1.0, -1.0);
  }

  CGContextDrawImage(UIGraphicsGetCurrentContext(), CGRectMake(0, 0, self.size.width, self.size.height), maskedImageRef);

  CGContextSetInterpolationQuality( UIGraphicsGetCurrentContext() , kCGInterpolationHigh );
    
  CGContextSetAllowsAntialiasing(UIGraphicsGetCurrentContext(), true);
  CGContextSetShouldAntialias(UIGraphicsGetCurrentContext(), true);
  UIImage *result = UIGraphicsGetImageFromCurrentImageContext();
  CGImageRelease(maskedImageRef);
  UIGraphicsEndImageContext();
    
  return result;
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods

//===============================================================================
//
//===============================================================================
+ (UIImage *)imageWithColor:(UIColor *)color
{
    UIImage *image = nil;
    
    CGColorSpaceRef colorSpace = CGColorSpaceCreateDeviceRGB();
    if(colorSpace!=NULL)
    {
        CGRect rect = CGRectMake(0, 0, 1, 1);
        
        CGContextRef context = CGBitmapContextCreate(NULL,
                                                     rect.size.width,
                                                     rect.size.height,
                                                     8,
                                                     [UIImage bytesPerRowWithWidth:rect.size.width bitCount:32],
                                                     colorSpace,
                                                     kCGBitmapAlphaInfoMask&kCGImageAlphaPremultipliedLast);
        if(context!=NULL)
        {
            CGContextSetFillColorWithColor(context, [color CGColor]);
            CGContextFillRect(context, rect);
            
            //////////////////////////////////////////////////
            
            CGImageRef cgImage = CGBitmapContextCreateImage(context);
            if(cgImage!=NULL)
            {
                image = [UIImage imageWithCGImage:cgImage];
                
                CGImageRelease(cgImage);
            }
            
            //////////////////////////////////////////////////
            
            CGContextRelease(context);
        }
        
        CGColorSpaceRelease(colorSpace);
    }
    
    return image;
}


//================================================================================
//
//================================================================================
+ (UIImage *)imageWithColor:(UIColor *)color size:(CGSize)size
{
    UIImage *image = [UIImage imageWithColor:color];
    
    if (image == nil)
    {
        return nil;
    }
    
    return [image imageExpandedWithFixedCornerCap:0.0 newSize:size];
}


//===============================================================================
//
//===============================================================================
+ (UIImage *)imageWithName:(NSString *)name
{
    UIImage *image = nil;
    NSString *filePath = nil;
    NSArray *nameArray = [name componentsSeparatedByString:@"."];
    
    if([nameArray count] == 1)
    {
        filePath = [[NSBundle mainBundle] pathForResource:name ofType:UIImageResourceType_PNG];

    }
    else if([nameArray count] == 2)
    {
        filePath = [[NSBundle mainBundle] pathForResource:[nameArray objectAtIndex:0] ofType:[nameArray objectAtIndex:1]];
        
    }

    if([filePath length])
    {
        image = [UIImage imageWithContentsOfFile:filePath];
    }
    
    return image;
}


//===============================================================================
//
//===============================================================================
+ (UIImage *)imageWithName:(NSString *)name forState:(UIControlState)state
{
    UIImage *image = nil;
    
    switch(state)
    {
        case UIControlStateSelected:
        {
            image = [UIImage imageWithName:[name stringByAppendingString:UIImageState_Selected]];
            break;
        }
        case UIControlStateDisabled:
        {
            image = [UIImage imageWithName:[name stringByAppendingString:UIImageState_Disabled]];
            break;
        }
        case UIControlStateHighlighted:
        {
            image = [UIImage imageWithName:[name stringByAppendingString:UIImageState_Highlighted]];
            break;
        }
        case UIControlStateNormal:
        default:
        {
            image = [UIImage imageWithName:[name stringByAppendingString:UIImageState_Normal]];
            break;
        }
    }
    
    return image;
}

//===============================================================================
//
//===============================================================================
+ (UIImage *)imageWithName:(NSString *)name forState:(UIControlState)state resizableCapInsets:(UIEdgeInsets)capInsets
{
    UIImage *image = [self imageWithName:name forState:state];
    
    if(image!=nil && UIEdgeInsetsEqualToEdgeInsets(capInsets, UIEdgeInsetsZero)!=YES)
    {
        image = [image resizableImageWithCapInsets:capInsets];
    }
    
    return image;
}

//===============================================================================
//
//===============================================================================
+ (NSUInteger)bytesPerRowWithWidth:(NSUInteger)width bitCount:(NSUInteger)bitCount
{
    return ((((bitCount*width)+31UL)>>5)<<2);
}


@end
