﻿using System;
using System.IO;
using Microsoft.Win32;
using PPScanControl;

namespace ScanControl_ConsoleSample
{
    public class Program
    {
        private const string SUB_KEY_PATH = @"SOFTWARE\PenPower\ScannerManager";
        private static ScanControl scanControl;

        public static void Main(string[] args)
        {
            // Set SDK LogLevel to 4 in registry path "HKEY_CURRENT_USER\SOFTWARE\PenPower\ScannerManager"
            // Log folder is in "%LocalAppData%\PenPower\ScannerManager"
            //WriteLogLevel(4);

            // Test external logger
            MyLogger myLogger = new MyLogger();

            // Prepare SDK and detect device
            scanControl = new ScanControl(myLogger);
            WriteLog("Detecting...");
            ScannerType scannerType = scanControl.GetConnectedScanner();
            if (scannerType == ScannerType.None)
            {
                WriteLog("No scanner detected.");
                Uninit();
                return;
            }
            WriteLog("Detected scanner: " + scannerType);

            // Open device
            WriteLog("Opening...");
            ReturnCode result = scanControl.Open(scannerType);
            if (result != ReturnCode.Success)
            {
                WriteLog("Open failed: " + result);
                Uninit();
                return;
            }
            WriteLog("Open success.");

            // Scan loop
            bool testAgain = true;
            do
            {
                TestScan();

                // Ask for another scan, or leave
                WriteLog("Scan another card? (Y/N)");
                ConsoleKeyInfo key = Console.ReadKey();
                if (key.Key != ConsoleKey.Y)
                    testAgain = false;
                Console.WriteLine();
            } while (testAgain);

            // Set SDK LogLevel to 2 and close SDK
            Uninit();
        }

        private static void TestScan()
        {
            WriteLog("Scanning...");

            // Sample scanning arguments
            int dpi = 300;
            ImageType type = ImageType.PNG;
            string directory = Environment.GetFolderPath(Environment.SpecialFolder.MyPictures);
            string fileName = DateTime.Now.ToString("yyyyMMdd-HHmmss", System.Globalization.CultureInfo.InvariantCulture) + ".png";
            fileName = Path.Combine(directory, "PPScanner", fileName);

            ScanCompletedEventArgs scanResult = scanControl.StartScanBlocking(dpi, type, fileName);
            if (scanResult.Result == ReturnCode.Success)
            {
                WriteLog("Scan success.");

                // if using image files
                if (scanResult.FileName != null)
                {
                    for (int i = 0; i < scanResult.FileName.Length; i++)
                    {
                        string log = $"File name [{i}]: {scanResult.FileName[i]}";
                        WriteLog(log);
                    }
                }
                else
                    WriteLog("File name: null");

                // if using base64 strings
                if (scanResult.ImageData != null)
                {
                    for (int i = 0; i < scanResult.ImageData.Length; i++)
                    {
                        string log = $"Image data [{i}]: ";
                        if (string.IsNullOrEmpty(scanResult.ImageData[i]))
                            log += "null";
                        else
                            log += scanResult.ImageData[i].Substring(0, 30) + "...";
                        WriteLog(log);
                    }
                }
                else
                    WriteLog("Image data: null");
            }
            else
            {
                WriteLog("Scan failed: " + scanResult.Result);
            }
        }

        private static void Uninit()
        {
            WriteLog("Closing...");
            try
            {
                scanControl.Close();
                scanControl = null;
                WriteLog("Close success.");
            }
            catch (Exception ex)
            {
                WriteLog("Close error: " + ex);
            }
            finally
            {
                //WriteLogLevel(2);
            }
            WriteLog("Press any key to exit...");
            Console.ReadKey();
            WriteLog("Bye");
        }

        private static void WriteLog(string log)
        {
            log = DateTime.Now.ToString("[HH:mm:ss.fff]", System.Globalization.CultureInfo.InvariantCulture) + " " + log;
            Console.WriteLine(log);
        }

        public static bool WriteLogLevel(int level)
        {
            // Log level: 1 (error only) ~ 4 (most verbose)
            int llValue = level;
            if (llValue > 4)
                llValue = 4;
            if (llValue < 1)
                llValue = 1;
            try
            {
                RegistryKey key = Registry.CurrentUser.OpenSubKey(SUB_KEY_PATH, true);
                if (key != null)
                {
                    key.SetValue("LogLevel", llValue);
                    key.Close();
                    key.Dispose();
                    key = null;
                    WriteLog("Set LogLevel to " + llValue);
                }
                else
                {
                    WriteLog("Create key: " + SUB_KEY_PATH);
                    key = Registry.CurrentUser.CreateSubKey(SUB_KEY_PATH, RegistryKeyPermissionCheck.ReadWriteSubTree);
                    if (key != null)
                    {
                        key.SetValue("LogLevel", llValue);
                        key.Close();
                        key.Dispose();
                        key = null;
                        WriteLog("Set LogLevel to " + llValue);
                    }
                    else
                        return false;
                }
            }
            catch { return false; }
            return true;
        }
    }

    public class MyLogger : ILogger
    {
        private const int logLevel = 4;

        public void WriteLog(int level, string log)
        {
            if (level > logLevel)
                return;
            Console.WriteLine($"\t[{level}] {log}");
        }
    }
}
