//
//  PPIDCardCaptureView.h
//

#import <UIKit/UIKit.h>
#import <PPIDCardSDK/PPIDCardDetectStatus.h>
#import <PPIDCardSDK/PPIDCardType.h>
#import <PPIDCardSDK/PPIDCardSide.h>

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mode setting (標準版要把mode都關掉)

// 只顯示preview畫面，所有狀態透過delegate通知。
#define _MODE_SINGLE_AUTO_

// 5合1偵測
//#define _MODE_MULTI_DETECT_

////////////////////////////////////////////////////////////////////////////////////////////////////

// local端辨識
//#define _LOCAL_RECOGNIZE_

////////////////////////////////////////////////////////////////////////////////////////////////////

@class PPIDCardCaptureView;
@class PPIDCardAppearanceModel;
@class PPIDCardResultModel;
@class PPDriverLicenseResultModel;
@class PPPassportResultModel;

@protocol PPIDCardCaptureViewDelegate <NSObject>
@optional

#if defined (_MODE_SINGLE_AUTO_)

/// 一般偵測通知
/// @param status 狀態
/// @param cornerPoints 身分證件四個端點座標
/// @note status狀態說明 \n
///  [01] PPIDCardDetectStatus_Start : 開始偵測 \n
///  [02] PPIDCardDetectStatus_RotateToLandscape : 請將手機轉至橫向 (不允許直式拍照版本才會回傳) \n
///  [03] PPIDCardDetectStatus_NoLicense : 沒有授權 \n
///  [04] PPIDCardDetectStatus_FailedToDetectEdge : 偵測不到四邊 \n
///  [05] PPIDCardDetectStatus_ImageTooSmall : 影像太小 \n
///  [06] PPIDCardDetectStatus_FailedToAdjustImage : 影像校正失敗 \n
///  [07] PPIDCardDetectStatus_FailedToDetectFace : 人臉偵測失敗 (偵測駕照才會回傳) \n
///  [08] PPIDCardDetectStatus_ImageBlur : 影像模糊 (有反光模糊偵測版本才會回傳) \n
///  [09] PPIDCardDetectStatus_ImageReflection : 影像反光 (有反光模糊偵測版本才會回傳) \n
///  [10] PPIDCardDetectStatus_TurnCardFront : 請翻到正面 (有正反面偵測版本才會回傳) \n
///  [11] PPIDCardDetectStatus_TurnCardBack : 請翻到反面 (有正反面偵測版本才會回傳) \n
///  [12] PPIDCardDetectStatus_FailedToRecognize : 辨識失敗 \n
///  [13] PPIDCardDetectStatus_ReadyToCapture : 準備拍照
- (void)captureViewDidDetectWithStatus:(PPIDCardDetectStatus)status cornerPoints:(NSArray<NSValue *> *)cornerPoints;


#ifdef _LOCAL_RECOGNIZE_

/// 身份證/健保卡辨識完成通知
/// @param result 辨識結果
- (void)captureViewDidRecognizeWithIDCardResult:(PPIDCardResultModel *)result;

/// 駕照辨識完成通知
/// @param result 辨識結果
- (void)captureViewDidRecognizeWithDriverLicenseResult:(PPDriverLicenseResultModel *)result;

/// 護照辨識完成通知
/// @param result 辨識結果
- (void)captureViewDidRecognizeWithPassportResult:(PPPassportResultModel *)result;

#else

/// 拍攝通知
/// @param jpgData 拍攝影像
- (void)captureViewDidCaptureWithJpgData:(NSData *)jpgData;

#endif // endof _LOCAL_RECOGNIZE_


#elif defined (_MODE_MULTI_DETECT_)

/// 按下cancel按鈕通知
- (void)captureViewDidClickCancel;

/// 拍照取得影像
/// @param frontSideImage 正面影像
/// @param backSideImage 反面影像
- (void)captureViewDidCaptureFrontSideImage:(UIImage *)frontSideImage backSideImage:(UIImage *)backSideImage;

#else

/// 按下cancel按鈕通知
- (void)captureViewDidClickCancel;

/// 辨識完成通知（標準流程）
/// @param frontSideResult 正面辨識結果
/// @param backSideResult 反面辨識結果
- (void)captureViewDidRecognizeWithFrontSideResult:(PPIDCardResultModel *)frontSideResult
                                    backSideResult:(PPIDCardResultModel *)backSideResult;

#endif

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPIDCardCaptureView : UIView

/// 代理
@property (atomic, assign) id<PPIDCardCaptureViewDelegate> delegate;

/// 證件種類，預設為身份證。
@property (nonatomic, assign) PPIDCardType cardType;

/// 駕照是否開啟人臉偵測，預設為開啟。
@property (nonatomic, assign) BOOL enableFaceDetection;

/// PPIDCardCaptureView initialize method
/// @param frame View frame
/// @param signature Signature from Penpower
- (instancetype)initWithFrame:(CGRect)frame
                    signature:(NSString *)signature;

@end


////////////////////////////////////////////////////////////////////////////////////////////////////


#if defined (_MODE_SINGLE_AUTO_)

@interface PPIDCardCaptureView (SingleAuto)

/// 偵測到正面或反面時進行拍照
@property (nonatomic, assign) PPIDCardSide captureSide;

/// camera影像顯示方向，預設為UIInterfaceOrientationPortrait。當parentView方向改變時才需調整。
@property (nonatomic, assign) UIInterfaceOrientation previewOrientation;

/// 切換到前鏡頭
/// @param toFrontCamera YES前鏡頭；NO後鏡頭
/// @note 前鏡頭拍照時直接回傳影像，不會進行任何偵測處理。 \n
/// 切換鏡頭時，系統會將手電筒關閉。
- (void)switchToFrontCamera:(BOOL)toFrontCamera;

/// 手電筒開關
/// @param turnOn 開啟
- (void)torchTurnOn:(BOOL)turnOn;

/// 對焦
/// @param viewPoint 對焦點座標 (PPIDCardCaptureView座標系)
- (void)focusWithPoint:(CGPoint)viewPoint;

/// 手動拍照
/// @note 即使在偵測過程中也可以進行拍照
- (void)capture;

/// 重新開始偵測
- (void)restart;

/// 設定外框
/// @param color 顏色
/// @param width 線寬
/// @param cornerRadius 圓角
- (void)borderWithColor:(UIColor *)color width:(CGFloat)width cornerRadius:(CGFloat)cornerRadius;

@end

#endif


////////////////////////////////////////////////////////////////////////////////////////////////////


#if defined (_MODE_MULTI_DETECT_)

@interface PPIDCardCaptureView (MultiDetect)

/// 介面開始時使用手動拍照模式（預設為NO，自動拍照模式）
@property (nonatomic, assign) BOOL defaultManualCapture;

/// 開啟正反面偵測
@property (nonatomic, assign) BOOL enableSideDetection;

/// 顯示樣式
@property (nonatomic, retain) PPIDCardAppearanceModel *appearanceModel;

@end

#endif
